# -*- coding: utf-8 -*-
from __future__ import unicode_literals

from django.contrib.auth import authenticate
from django.core.paginator import Paginator, PageNotAnInteger, EmptyPage
from django.db import IntegrityError
from django.db.models import Q
from rest_framework.authentication import TokenAuthentication
from rest_framework.authtoken.models import Token
from rest_framework.permissions import IsAuthenticated
from rest_framework.views import APIView

from applications.users.models import User
from applications.utilities.common import raise_error, response, ParameterError
from applications.users.serializers import SearchUserSerializer
from applications.utilities.permissions import IsICarAdmin
from iCar.constants import NUM_OF_PAGINATION_ITEMS


class LoginView(APIView):
    def post(self, request, *args, **kwargs):
        """
        login
        --------

        parameters
            - name: username
            type: string
            description: username e kArbar

            - name: password
            type: string
            description: password e kArbar

        Response Messages
            - code: 200
            response: {"token"}

        Error Messages
            - code: 001
            response: {"status": "not valid parameter", "code": "001"}

            - code: 002
            response: {"status":"object not found","code":"002"}
        """
        try:
            username = request.data['username']
            password = request.data['password']
        except ParameterError:
            return raise_error("PARAMETERS_ERROR")

        user = authenticate(username=username, password=password)
        if user and user.is_admin:
            try:
                token = Token.objects.get(user=user)
            except Token.DoesNotExist:
                token = Token.objects.create(user=user)
            return response({"token": token.key})
        else:
            return raise_error("NOT_FOUND_ERROR")


class UserSearchView(APIView):
    serializer_class = SearchUserSerializer
    permission_classes = (IsAuthenticated, IsICarAdmin)
    authentication_classes = (TokenAuthentication,)

    def post(self, request, *args, **kwargs):
        """
        search in users list with pagination
        --------
        parameters
            - name: code_melli
            type: string
            description: code melli e kArbar
            optional: True

            - name: name
            type: string
            description: esm ya famil e kArbar
            optional: True

        Response Messages
            - code: 200
            response: {
                "num_of_pages": 1,
                "users": [
                    {
                        "id",
                        "first_name": "عباس",
                        "last_name": "علی خانی",
                        "code_melli": "hosein",
                        "mobile_number": "0987653",
                        "father": "کریم",
                        "shomare_shenasname": "45678",
                        "address": "",
                        "sadere": "بابل",
                        "phone_number": "12345678",
                        "code_posti": "34567"
                    },
                    {
                        "id",
                        "first_name": "علی",
                        "last_name": "محمودی",
                        "code_melli": "ali",
                        "mobile_number": "098765412",
                        "father": "حسین",
                        "shomare_shenasname": "21234567845",
                        "address": "",
                        "sadere": "تهران",
                        "phone_number": "8765434567",
                        "code_posti": "12345678"
                    }
                ]
            }

        """
        name = request.data.get('name', None)
        username = request.data.get('code_melli', None)
        page = request.query_params.get("page")

        users = User.objects.all().order_by("-date_joined")

        if name:
            users = users.filter(Q(first_name__contains=name) | Q(last_name__contains=name))
        if username:
            users = users.filter(username__contains=username)

        paginator = Paginator(users, NUM_OF_PAGINATION_ITEMS)

        try:
            users = paginator.page(page)
        except PageNotAnInteger:
            users = paginator.page(1)
        except EmptyPage:
            users = []
        serializers = self.serializer_class(users, many=True).data
        return response({"users": serializers, "num_of_pages": paginator.num_pages})


class UserView(APIView):
    permission_classes = (IsAuthenticated, IsICarAdmin)
    authentication_classes = (TokenAuthentication,)

    def post(self, request):
        """
        create new user
        --------
        parameters
            - name: code_melli
            type: string
            description: code melli e kArbar

            - name: first_name
            type: string
            description: esm e kArbar

            - name: last_name
            type: string
            description: family e kArbar

            - name: mobile_number
            type: string
            description: shomare mobile

            - name: address
            type: string
            description: address

            - name: father
            type: string
            description: esm e pedar

            - name: shomare_shenasname
            type: string
            description: shomare shenAsnAme

            - name: sadere
            type: string
            description: sAdere

            - name: phone_number
            type: string
            description: shomare telefon

            - name: code_posti
            type: string
            description: code e posti

        Response Messages
            - code: 200
            response: {"status": "ok"}

        Error Messages
            - code: 001
            response: {"status": "not valid parameter", "code": "001"}

            - code: 005
            response: {"status": "You can do this just one time", "code": "005"}

        """
        try:
            username = request.data["code_melli"]
            first_name = request.data["first_name"]
            last_name = request.data["last_name"]
            mobile_number = request.data["mobile_number"]
            address = request.data["address"]
            father = request.data["father"]
            shomare_shenasname = request.data["shomare_shenasname"]
            sadere = request.data["sadere"]
            phone_number = request.data["phone_number"]
            code_posti = request.data["code_posti"]
        except ParameterError:
            return raise_error("PARAMETERS_ERROR")

        try:
            User.objects.create(
                username=username,
                first_name=first_name,
                last_name=last_name,
                mobile_number=mobile_number,
                address=address,
                father=father,
                shomare_shenasname=shomare_shenasname,
                sadere=sadere,
                phone_number=phone_number,
                code_posti=code_posti
            )
        except IntegrityError:
            return raise_error("JUST_ONE_TIME_ERROR")

        return response({"status": "ok"})



class GetUserView(APIView):
    permission_classes = (IsAuthenticated, IsICarAdmin)
    authentication_classes = (TokenAuthentication,)
    serializer_class = SearchUserSerializer

    def get(self, request, **kwargs):
        """
        get a user
        --------
        parameters
            - name: id
            type: integer
            description: id e user e marbo0te

        Response Messages
            - code: 200
            response: {
                "id": 5,
                "first_name": "نتاز",
                "last_name": "هتاذ",
                "code_melli": "98765432",
                "mobile_number": "34567890",
                "father": "قفاتپو",
                "shomare_shenasname": "3456789",
                "address": "نلفغعتدذرب",
                "sadere": "خهنتارزبفغات",
                "phone_number": "98765",
                "code_posti": "4567"
            }

        Error Messages
            - code: 001
            response: {"status": "not valid parameter", "code": "001"}

            - code: 002
            response: {
                "status": "object not found",
                "field": "User",
                "code": "002"
            }

        """
        user_id = kwargs["id"]

        try:
            user = User.objects.get(id=user_id)
        except User.DoesNotExist:
            return raise_error("NOT_FOUND_ERROR", "User")

        return response(self.serializer_class(user).data)

    def put(self, request, **kwargs):
        """
        edit a user
        --------
        parameters
            - name: id
            type: integer
            description: id e user e marbo0te

            - name: first_name
            type: string
            description: esm e kArbar

            - name: last_name
            type: string
            description: family e kArbar

            - name: mobile_number
            type: string
            description: shomare mobile

            - name: address
            type: string
            description: address

            - name: father
            type: string
            description: esm e pedar

            - name: shomare_shenasname
            type: string
            description: shomare shenAsnAme

            - name: sadere
            type: string
            description: sAdere

            - name: phone_number
            type: string
            description: shomare telefon

            - name: code_posti
            type: string
            description: code e posti

        Response Messages
            - code: 200
            response: {"status": "ok"}

        Error Messages
            - code: 001
            response: {"status": "not valid parameter", "code": "001"}

            - code: 002
            response: {"status":"object not found","code":"002"}

        """
        try:
            user_id = kwargs["id"]
            first_name = request.data.get("first_name", None)
            last_name = request.data.get("last_name", None)
            mobile_number = request.data.get("mobile_number", None)
            address = request.data.get("address", None)
            father = request.data.get("father", None)
            shomare_shenasname = request.data.get("shomare_shenasname", None)
            sadere = request.data.get("sadere", None)
            phone_number = request.data.get("phone_number", None)
            code_posti = request.data.get("code_posti", None)
        except ParameterError:
            return raise_error("PARAMETERS_ERROR")

        try:
            user = User.objects.get(id=user_id)
        except User.DoesNotExist:
            return raise_error("NOT_FOUND_ERROR")

        user.update(
            first_name=first_name,
            last_name=last_name,
            mobile_number=mobile_number,
            address=address,
            father=father,
            shomare_shenasname=shomare_shenasname,
            sadere=sadere,
            phone_number=phone_number,
            code_posti=code_posti
        )

        return response({"status": "ok"})
