# -*- coding: utf-8 -*-
from __future__ import unicode_literals

import jdatetime
from rest_framework.authentication import TokenAuthentication
from rest_framework.permissions import IsAuthenticated
from rest_framework.views import APIView

from applications.cars.serializers import ShowCarSerializer
from applications.configs.models import GholnameSettings
from applications.gholname.models import LizingGholname, NaghdiGholname, AghsatiGholname, Gholname
from applications.gholname.serializers import GholnameSerializer
from applications.users.models import User
from applications.users.serializers import UserInGholnameSerializer
from applications.utilities.common import response, raise_error, RequestParser, ParameterError, EmamParameterError
from applications.utilities.permissions import IsICarAdmin


class ShowSefrLizingView(APIView):
    permission_classes = (IsAuthenticated, IsICarAdmin)
    authentication_classes = (TokenAuthentication,)

    def get(self, request, **kwargs):
        """
        daryAft e gholnAme lizingi
        --------
        parameters
            - name: gholname_id
            type: integer

        Response Messages
            - code:200
            agar gholnAme sefr bAshe, yek field be onvAn e "manba_kharid" miad to0 response be in so0rat:

            "manba_kharid" :
            {
                "first_name": "مجید",
                "last_name": "لعل",
                "father": "م",
                "shomare_shenasname": "م",
                "username": "م",
                "sadere": "م",
                "phone_number": "م",
                "mobile_number": "م",
                "code_posti": "م",
                "address": "م"
            }
            response:{
                "car": {
                    "name": "پورشه",
                    "system": "kado0tanbal",
                    "tip": "خوشتیپ",
                    "model": "بالا",
                    "color": "یشمی",
                    "pelak": "123ب21",
                    "shasi": "2345",
                    "motor": "ضعیف"
                },
                "baghimande": 95000,
                "gholname": {
                    "main_cost": 10000,
                    "id",
                    "foro0shande": {
                        "first_name": "علی",
                        "last_name": "محمودی",
                        "father": "حسین",
                        "shomare_shenasname": "21234567845",
                        "username": "ali",
                        "sadere": "تهران",
                        "phone_number": "8765434567",
                        "mobile_number": "098765412",
                        "code_posti": "12345678",
                        "address": "",
                        "faskh_info": {"faskh_datetime", "sabt_datetime", "description"}
                    },
                    "kharidar": {
                        "first_name": "علی",
                        "last_name": "محمودی",
                        "father": "حسین",
                        "shomare_shenasname": "21234567845",
                        "username": "ali",
                        "sadere": "تهران",
                        "phone_number": "8765434567",
                        "mobile_number": "098765412",
                        "code_posti": "12345678",
                        "address": ""
                    },
                    "shomare_sanad": 10013,
                    "datetime": "1525693505",
                    "dong": "شش",
                    "total_cost": 100000,
                    "pish_pardakht": 5000,
                    "kharj_sanad": 10000,
                    "tahvil_datetime": "1525693505",
                    "description": "بسیار زیبا"
                },
                "chek_num":10,
                "mablagh_chek ":200000,
                "sharayet": "شرایط خوب\r\nشرایط بد"
            }

        Error Messages
            - code: 001
            response: {"status": "not valid parameter", "code": "001"}

            - code: 002
            response: {"status":"object not found","code":"002"}

        """

        gholname_id = int(kwargs["id"])

        try:
            gholname_lizing = LizingGholname.objects.get(gholname_id=gholname_id)
            gholname = gholname_lizing.gholname
        except LizingGholname.DoesNotExist:
            return raise_error("NOT_FOUND_ERROR")

        baghimande = gholname.total_cost - gholname.pish_pardakht

        try:
            gholname_settings = GholnameSettings.objects.all()[0]
        except IndexError:
            gholname_settings = GholnameSettings.objects.create()

        kharj_sanad = ""
        if int(gholname.kharj_sanad) == 0:
            kharj_sanad = "خریدار"
        elif int(gholname.kharj_sanad) == 1:
            kharj_sanad = "فروشنده"
        elif int(gholname.kharj_sanad) == 2:
            kharj_sanad = "هر دو"

        jalalian_tahvil_datetime = jdatetime.date.fromgregorian(date=gholname.tahvil_datetime.date())
        date = "{}:{} ، {}/{}/{}".format(gholname.tahvil_datetime.hour, gholname.tahvil_datetime.minute,
                                         jalalian_tahvil_datetime.year, jalalian_tahvil_datetime.month,
                                         jalalian_tahvil_datetime.day)

        serializer = {
            "gholname": GholnameSerializer(gholname).data,
            "chek_num": gholname_lizing.chek_num,
            "moddat_chek": gholname_lizing.moddat_chek,
            "car": ShowCarSerializer(gholname.car).data,
            "baghimande": baghimande,
            "sharayet": gholname_settings.sharayet_moamele_lizingi.replace("%x", kharj_sanad).replace("%y", date),
        }

        gholname_kharid = gholname.foro0sh.gholname_kharid

        if gholname.is_sefr:
            serializer["manba_kharid"] = UserInGholnameSerializer(gholname_kharid.foro0shande).data

        return response(
            serializer
        )

    def put(self, request, **kwargs):
        """
        virAyesh e gholnAme lizingi
        --------
        parameters
            -name: id | in url
            type: int
            description: id e gholnAme

            - name: name
            type: string | no e khodro

            - name: system
            type: string

            - name: tip
            type: string

            - name: model
            type: string

            - name: color
            type: string

            - name: pelak
            type: string

            - name: shasi
            type: string

            - name: motor
            type: string

            - name: saheb_pelak_id
            type: int
            optional: True

            - name: code
            type: string
            optional: True

            - name: VIN
            type: string
            optional: True

            - name: gholname_datetime
            type: int

            - name: tahvil_datetime
            type: int

            - name: total_cost
            type: int

            - name: pish_pardakht
            type: int

            - name: kharj_sanad
            type: int

            - name: dong
            type: string

            - name: kharidar_id
            type: int

            - name: foro0shande_id
            type: int

            - name: description
            type: string

        Response Messages
            - code:200
            response:{"status": "ok"}

        Error Messages
            - code: 001
            response: {"status": "not valid parameter", "code": "001"}

            - code: 002
            response: {"status":"object not found","code":"002"}

        """

        parser = RequestParser(request.data)

        gholname_id = int(kwargs["id"])

        try:
            gholname_lizing = LizingGholname.objects.get(gholname_id=gholname_id)
            gholname = gholname_lizing.gholname
        except LizingGholname.DoesNotExist:
            return raise_error("NOT_FOUND_ERROR")

        try:
            name = parser.str("name", default=None)
            system = parser.str("system", default=None)
            tip = parser.str("tip", default=None)
            model = parser.str("model", default=None)
            color = parser.str("color", default=None)
            pelak = parser.str("pelak", default=None)
            shasi = parser.str("shasi", default=None)
            motor = parser.str("motor", default=None)
            gholname_datetime = parser.timestamp("gholname_datetime", default=None)
            tahvil_datetime = parser.timestamp("tahvil_datetime", default=None)
            total_cost = parser.int("total_cost", default=None)
            pish_pardakht = parser.int("pish_pardakht", default=None)
            kharj_sanad = parser.int("kharj_sanad", default=None)
            dong = parser.str("dong", default=None)
            kharidar_id = parser.int("kharidar_id", default=None)
            manba_kharid_id = parser.int("manba_kharid_id", default=None)
            foro0shande_id = parser.int("foro0shande_id", default=None)
            description = parser.str("description", default=None)
            chek_num = parser.int("chek_num", default=None)
            moddat_chek = parser.int("moddat_chek", default=None)
        except (KeyError, ValueError):
            return raise_error("PARAMETERS_ERROR")

        is_pelaked = False
        try:
            saheb_pelak_id = int(request.data["saheb_pelak_id"])
            saheb_pelak = User.objects.get(id=saheb_pelak_id)
            is_pelaked = True
        except (ParameterError, User.DoesNotExist):
            saheb_pelak = None

        code = request.data.get("code", "")
        vin = request.data.get("VIN", "")

        if kharidar_id:
            try:
                kharidar = User.objects.get(id=kharidar_id)
            except User.DoesNotExist:
                return raise_error("NOT_FOUND_ERROR")
        else:
            kharidar = None

        if foro0shande_id:
            try:
                foro0shande = User.objects.get(id=foro0shande_id)
            except User.DoesNotExist:
                return raise_error("NOT_FOUND_ERROR")
        else:
            foro0shande = None

        if manba_kharid_id:
            try:
                manba_kharid = User.objects.get(id=manba_kharid_id)
            except User.DoesNotExist:
                return raise_error("NOT_FOUND_ERROR")
        else:
            manba_kharid = None

        car_properties = {"name": name, "system": system, "tip": tip, "model": model, "color": color,
                          "pelak": pelak,
                          "shasi": shasi, "motor": motor}
        if code:
            car_properties["code"] = code
        if vin:
            car_properties["vin"] = vin

        if is_pelaked:
            car_properties["saheb_pelak"] = saheb_pelak

        gholname.car.update(**car_properties)

        gholname.update(
            description=description or '',
            kharidar=kharidar,
            foro0shande=foro0shande,
            datetime=gholname_datetime,
            tahvil_datetime=tahvil_datetime,
            total_cost=total_cost,
            pish_pardakht=pish_pardakht,
            kharj_sanad=kharj_sanad,
            dong=dong,
        )

        kharid_namayeshgah_gholname = gholname.foro0sh.gholname_kharid

        kharid_namayeshgah_gholname.update(
            description=description or '',
            kharidar=foro0shande,
            foro0shande=manba_kharid,
            datetime=gholname_datetime,
            tahvil_datetime=tahvil_datetime,
            total_cost=total_cost,
            pish_pardakht=pish_pardakht,
            kharj_sanad=kharj_sanad,
            dong=dong
        )

        gholname_lizing.update(
            chek_num=chek_num,
            moddat_chek=moddat_chek
        )

        return response(
            {
                "status": "ok"
            }
        )


class EditSefrNaghdiView(APIView):
    permission_classes = (IsAuthenticated, IsICarAdmin)
    authentication_classes = (TokenAuthentication,)

    def put(self, request, **kwargs):
        """
        virAyesh e gholnAme naghdi e sefr
        --------
        parameters
            -name: id | in url
            type: int
            description: id e gholnAme

            - name: name
            type: string | no e khodro

            - name: system
            type: string

            - name: tip
            type: string

            - name: model
            type: string

            - name: color
            type: string

            - name: pelak
            type: string

            - name: shasi
            type: string

            - name: motor
            type: string

            - name: gholname_datetime
            type: int

            - name: tahvil_datetime
            type: int

            - name: total_cost
            type: int

            - name: pish_pardakht
            type: int

            - name: kharj_sanad
            type: int

            - name: saheb_pelak_id
            type: int
            optional: True

            - name: dong
            type: string

            - name: kharidar_id
            type: int

            - name: manba_kharid_id
            type: int

            - name: foro0shande_id
            type: int

            - name: description
            type: string

            - name: shomare_dasti
            type: int

            - name: code
            type: string
            optional: True

            - name: VIN
            type: string
            optional: True

        Response Messages
            - code:200
            response:{"status": "ok"}

        Error Messages
            - code: 001
            response: {"status": "not valid parameter", "code": "001"}

            - code: 002
            response: {"status":"object not found","code":"002"}

        """
        parser = RequestParser(request.data)

        gholname_id = int(kwargs["id"])

        try:
            gholname_naghdi = NaghdiGholname.objects.get(gholname_id=gholname_id)
            gholname = gholname_naghdi.gholname
        except NaghdiGholname.DoesNotExist:
            return raise_error("NOT_FOUND_ERROR")

        try:
            name = parser.str("name", default=None)
            system = parser.str("system", default=None)
            tip = parser.str("tip", default=None)
            model = parser.str("model", default=None)
            color = parser.str("color", default=None)
            pelak = parser.str("pelak", default=None)
            shasi = parser.str("shasi", default=None)
            motor = parser.str("motor", default=None)
            shomare_dasti = int(request.data["shomare_dasti"])
            gholname_datetime = parser.timestamp("gholname_datetime", default=None)
            tahvil_datetime = parser.timestamp("tahvil_datetime", default=None)
            total_cost = parser.int("total_cost", default=None)
            pish_pardakht = parser.int("pish_pardakht", default=None)
            kharj_sanad = parser.int("kharj_sanad", default=None)
            dong = parser.str("dong", default=None)
            kharidar_id = parser.int("kharidar_id", default=None)
            foro0shande_id = parser.int("foro0shande_id", default=None)
            manba_kharid_id = parser.int("manba_kharid_id", default=None)
            description = parser.str("description", default=None)
        except (KeyError, ValueError):
            return raise_error("PARAMETERS_ERROR")

        is_pelaked = False
        try:
            saheb_pelak_id = int(request.data["saheb_pelak_id"])
            saheb_pelak = User.objects.get(id=saheb_pelak_id)
            is_pelaked = True
        except (ParameterError, User.DoesNotExist):
            saheb_pelak = None

        code = request.data.get('code', '')
        vin = request.data.get('VIN', '')
        if kharidar_id:
            try:
                kharidar = User.objects.get(id=kharidar_id)
            except User.DoesNotExist:
                return raise_error("NOT_FOUND_ERROR", "kharidar")
        else:
            kharidar = None

        if foro0shande_id:
            try:
                foro0shande = User.objects.get(id=foro0shande_id)
            except User.DoesNotExist:
                return raise_error("NOT_FOUND_ERROR", "foro0shande")
        else:
            foro0shande = None

        if manba_kharid_id:
            try:
                manba_kharid = User.objects.get(id=manba_kharid_id)
            except User.DoesNotExist:
                return raise_error("NOT_FOUND_ERROR", "manba_kharid")
        else:
            manba_kharid = None

        car_properties = {"name": name, "system": system, "tip": tip, "model": model, "color": color,
                          "pelak": pelak,
                          "shasi": shasi, "motor": motor}

        if is_pelaked:
            car_properties["saheb_pelak"] = saheb_pelak
        car_properties["code"] = code
        car_properties["vin"] = vin

        gholname.car.update(**car_properties)

        gholname.update(
            description=description or '',
            kharidar=kharidar,
            foro0shande=foro0shande,
            datetime=gholname_datetime,
            tahvil_datetime=tahvil_datetime,
            total_cost=total_cost,
            pish_pardakht=pish_pardakht,
            kharj_sanad=kharj_sanad,
            dong=dong,
            shomare_dasti=shomare_dasti
        )

        kharid_namayeshgah_gholname = gholname.foro0sh.gholname_kharid

        kharid_namayeshgah_gholname.update(
            description=description or '',
            kharidar=foro0shande,
            foro0shande=manba_kharid,
            datetime=gholname_datetime,
            tahvil_datetime=tahvil_datetime,
            total_cost=total_cost,
            pish_pardakht=pish_pardakht,
            kharj_sanad=kharj_sanad,
            dong=dong
        )

        return response(
            {
                "status": "ok"
            }
        )


class EditSefrAghsatiView(APIView):
    permission_classes = (IsAuthenticated, IsICarAdmin)
    authentication_classes = (TokenAuthentication,)

    def put(self, request, **kwargs):
        """
        virAyesh e gholnAme aghsAti e sefr
        --------
        parameters
            -name: id | in url
            type: int
            description: id e gholnAme

            - name: name
            type: string | no e khodro

            - name: system
            type: string

            - name: tip
            type: string

            - name: model
            type: string

            - name: color
            type: string

            - name: pelak
            type: string

            - name: shasi
            type: string

            - name: motor
            type: string

            - name: gholname_datetime
            type: int

            - name: tahvil_datetime
            type: int

            - name: total_cost
            type: int

            - name: saheb_pelak_id
            type: int
            optional: True

            - name: pish_pardakht
            type: int

            - name: ghest_num
            type: int
            description: tedAd e ghest

            - name: percent_num
            type: int
            description: darsad e so0d

            - name: ghest_start_datetime
            type: int
            description: tArikh e shoro0 e ghest

            - name: kharj_sanad
            type: int

            - name: dong
            type: string

            - name: kharidar_id
            type: int

            - name: manba_kharid_id
            type: int

            - name: foro0shande_id
            type: int

            - name: description
            type: string

            - name: shomare_dasti
            type: int

            - name: code
            type: string
            optional: True

            - name: VIN
            type: string
            optional: True

        Response Messages
            - code:200
            response:{"status": "ok"}

        Error Messages
            - code: 001
            response: {"status": "not valid parameter", "code": "001"}

            - code: 002
            response: {"status":"object not found","code":"002"}

        """
        parser = RequestParser(request.data)
        gholname_id = int(kwargs["id"])

        try:
            gholname_aghsati = AghsatiGholname.objects.get(gholname_id=gholname_id)
            gholname = gholname_aghsati.gholname
        except NaghdiGholname.DoesNotExist:
            return raise_error("NOT_FOUND_ERROR")

        try:
            name = parser.str("name", default=None)
            system = parser.str("system", default=None)
            tip = parser.str("tip", default=None)
            model = parser.str("model", default=None)
            color = parser.str("color", default=None)
            pelak = parser.str("pelak", default=None)
            shasi = parser.str("shasi", default=None)
            motor = parser.str("motor", default=None)
            shomare_dasti = parser.int("shomare_dasti", default=None)
            gholname_datetime = parser.timestamp("gholname_datetime", default=None)
            tahvil_datetime = parser.timestamp("tahvil_datetime", default=None)
            total_cost = parser.int("total_cost", default=None)
            pish_pardakht = parser.int("pish_pardakht", default=None)
            kharj_sanad = parser.int("kharj_sanad", default=None)
            dong = parser.str("dong", default=None)
            kharidar_id = parser.int("kharidar_id", default=None)
            foro0shande_id = parser.int("foro0shande_id", default=None)
            manba_kharid_id = parser.int("manba_kharid_id", default=None)
            description = parser.str("description", default=None)
            ghest_num = parser.int("ghest_num", default=None)
            percent = parser.float("percent_num", default=None)
            ghest_start_datetime = parser.timestamp("ghest_start_datetime", default=None)
        except EmamParameterError as e:
            return raise_error("PARAMETERS_ERROR", e.field_name)

        is_pelaked = False
        try:
            saheb_pelak_id = int(request.data["saheb_pelak_id"])
            saheb_pelak = User.objects.get(id=saheb_pelak_id)
            is_pelaked = True
        except (ParameterError, User.DoesNotExist):
            saheb_pelak = None

        code = request.data.get("code", "")
        vin = request.data.get("VIN", "")

        if kharidar_id:
            try:
                kharidar = User.objects.get(id=kharidar_id)
            except User.DoesNotExist:
                return raise_error("NOT_FOUND_ERROR", "kharidar")
        else:
            kharidar = None

        if foro0shande_id:
            try:
                foro0shande = User.objects.get(id=foro0shande_id)
            except User.DoesNotExist:
                return raise_error("NOT_FOUND_ERROR", "foro0shande")
        else:
            foro0shande = None

        if manba_kharid_id:
            try:
                manba_kharid = User.objects.get(id=manba_kharid_id)
            except User.DoesNotExist:
                return raise_error("NOT_FOUND_ERROR", "manba_kharid")
        else:
            manba_kharid = None

        car_properties = {"name": name, "system": system, "tip": tip, "model": model, "color": color,
                          "pelak": pelak,
                          "shasi": shasi, "motor": motor}
        if code:
            car_properties["code"] = code
        if vin:
            car_properties["vin"] = vin

        if is_pelaked:
            car_properties["saheb_pelak"] = saheb_pelak

        gholname.car.update(**car_properties)

        gholname.update(
            description=description or '',
            kharidar=kharidar,
            foro0shande=foro0shande,
            datetime=gholname_datetime,
            tahvil_datetime=tahvil_datetime,
            total_cost=total_cost,
            pish_pardakht=pish_pardakht,
            kharj_sanad=kharj_sanad,
            dong=dong,
            shomare_dasti=shomare_dasti
        )

        kharid_namayeshgah_gholname = gholname.foro0sh.gholname_kharid

        kharid_namayeshgah_gholname.update(
            description=description or '',
            kharidar=foro0shande,
            foro0shande=manba_kharid,
            datetime=gholname_datetime,
            tahvil_datetime=tahvil_datetime,
            total_cost=total_cost,
            pish_pardakht=pish_pardakht,
            kharj_sanad=kharj_sanad,
            dong=dong
        )

        gholname_aghsati.update(num_of_ghest=ghest_num, percent=percent, ghest_start_datetime=ghest_start_datetime)

        return response(
            {
                "status": "ok"
            }
        )


class CancelGholnameView(APIView):
    permission_classes = (IsAuthenticated, IsICarAdmin)
    authentication_classes = (TokenAuthentication,)

    def delete(self, request, **kwargs):
        """
        cancel kardan e yek gholnAme
        --------
        parameters
            - name: id
            type: integer
            description: id e gholnAme

        Response Messages
            - code:200
            response:{"status": "ok"}

        Error Messages

            - code: 002
            response: {
                "data": {},
                "error": {
                    "status": "object not found",
                    "field": "GHOLNAME",
                    "code": "002"
                }
            }
        """
        gholname_id = kwargs["id"]

        try:
            gholname = Gholname.objects.get(id=gholname_id)
        except Gholname.DoesNotExist:
            return raise_error("NOT_FOUND_ERROR", item="GHOLNAME")

        gholname.delete()

        return response({"status": "ok"})
