# -*- coding: utf-8 -*-
from __future__ import unicode_literals

from django.core.paginator import Paginator, PageNotAnInteger, EmptyPage
from django.db.models import Q
from rest_framework.authentication import TokenAuthentication
from rest_framework.permissions import IsAuthenticated
from rest_framework.views import APIView

from applications.baygani.models import KharidForo0shCar, SanadMotefarreghe, SanadMotefarregheCar, MadarekTahvili, Exit, \
    Enter, ShomareBaygani
from applications.baygani.serializers import BayganiListSerializer, ShowBayganiSerializer, \
    SanadMotefarregheListSerializer, SanadMotefarregheCarListSerializer, SanadMotefarregheCarSerializer
from applications.cars.models import CarSystem
from applications.users.models import User
from applications.utilities.common import RequestParser, EmamParameterError, raise_error, response
from applications.utilities.permissions import IsICarAdmin
from iCar.constants import NUM_OF_PAGINATION_ITEMS


class BayganiListView(APIView):
    serializer_class = BayganiListSerializer
    permission_classes = (IsAuthenticated, IsICarAdmin)
    authentication_classes = (TokenAuthentication,)

    def get(self, request):
        """
        daryAfte list e mavArede mojo0d dar baygani ba filtration
        --------
        parameters
            - name: page
            type: int
            description: shomAre pagination -> ?page=1

            - name: name
            type: string
            description: nAm e kharidAr ya foro0shande

            - name: code_melli
            type: string
            description: code melli e kharidAr ya foro0shande

            - name: shasi
            type: string
            description: shomare shasi e mashin

            - name: pelak_ghadim
            type: string
            description: pelak e ghadim e mashin

            - name: pelak_jadid
            type: string
            description: pelak e jadid e mashin

            - name: saheb_pelak_ghadim_id
            type: int
            description: id e saheb pelak e ghadim

            - name: vakil_id
            type: int
            description: id e vakil

            - name: khodro
            type: string
            description: nAme khodro

            - name: model
            type: string
            description: model e khodro

            - name: system
            type: string
            description: system e mashin

            - name: etmam_vekalat_foro0sh
            type: timestamp
            description: vAzehe dg

            - name: etmam_vekalat_kari
            type: timestamp
            description: inam vAzehe

        Response Messages
            - code: 200
            response: {
                "num_of_pages": 1,
                "baygani": [
                    {
                        "id": 1,
                        "shomare_baygani": 10,
                        "foro0shande": "یبلاتن بلاتن",
                        "kharid_namayeshgah_datetime": "1525677305",
                        "kharidar": "نتاز هتاذ",
                        "gholname_datetime": "1526446920",
                        "current_state": 2,
                        "vekalat_kari_engheza": "1527130993",
                        "sanad_dar_rahn_engheza": "1527208631",
                        "sanad_dar_rahn_vekalat_kari": false,
                        "vekalat_foro0sh_engheza": "1527208634",
                        "taviz_pelak_shode_state": 0,
                        "madarek_tahvili": {
                            "id": 2,
                            "avArez_ta": null,
                            "sanad_sabz": false,
                            "sanad_company": true,
                            "sanad_ghati": false,
                            "taiidie_pelak": false,
                            "vekalat_kari": false,
                            "vekalat_foro0sh": false
                        },
                        "tahvil_dahande": "حسین کریمی",
                        "description": "بسیار زیبا",
                        "pelak_ghadim": "asd",
                        "pelak_jadid": "asd",
                        "car": {
                            "name": "پورشه",
                            "model": "بالا",
                            "shasi": "2345",
                            "system": "kado0tanbal"
                        },
                        "saheb_pelak_ghadim": "یبلاتن بلاتن"
                    }
                ]
            }
        """
        parser = RequestParser(request.query_params)
        try:
            name = parser.str("name", None)
            shasi = parser.str("shasi", None)
            username = parser.str("code_melli", None)
            pelak_ghadim = parser.str("pelak_ghadim", None)
            pelak_jadid = parser.str("pelak_jadid", None)
            etmam_vekalat_kari = parser.timestamp("etmam_vekalat_kari", None)
            etmam_vekalat_foro0sh = parser.timestamp("etmam_vekalat_foro0sh", None)
            car_name = parser.str("khodro", None)
            model = parser.str("model", None)
            system = parser.str("system", None)
            vakil_id = parser.int("vakil_id", None)
            saheb_pelak_ghadim_id = parser.int("saheb_pelak_ghadim_id", None)
        except EmamParameterError as e:
            return raise_error("PARAMETERS_ERROR", e.field_name)

        filter_list = {"gholname_kharid__faskh__isnull": True, "gholname_foro0sh__faskh__isnull": True,
                       "is_baygani": True}

        if saheb_pelak_ghadim_id:
            filter_list["gholname_kharid__foro0shande__id"] = saheb_pelak_ghadim_id

        if pelak_ghadim:
            filter_list["pelak_ghadim__contains"] = pelak_ghadim

        if vakil_id:
            filter_list["vakil__id"] = vakil_id

        if pelak_jadid:
            filter_list["pelak_jadid__contains"] = pelak_jadid

        if car_name:
            filter_list["gholname_kharid__car__name__contains"] = car_name

        if model:
            filter_list["gholname_kharid__car__model__contains"] = model

        if system:
            filter_list["gholname_kharid__car__system__contains"] = system

        if etmam_vekalat_foro0sh:
            filter_list["vekalat_foro0sh_engheza__lte"] = etmam_vekalat_foro0sh

        if etmam_vekalat_kari:
            filter_list["vekalat_kari_engheza__lte"] = etmam_vekalat_kari

        if shasi:
            filter_list["gholname_kharid__car__shasi__contains"] = shasi

        kharid_foro0shs = KharidForo0shCar.objects.filter(**filter_list).order_by("-shomare_baygani__number")

        if name:
            kharid_foro0shs = kharid_foro0shs.filter(
                Q(gholname_foro0sh__kharidar__first_name__contains=name) |
                Q(gholname_foro0sh__kharidar__last_name__contains=name) |
                Q(gholname_kharid__foro0shande__first_name__contains=name) |
                Q(gholname_kharid__foro0shande__last_name__contains=name)
            )

        if username:
            kharid_foro0shs = kharid_foro0shs.filter(
                Q(gholname_foro0sh__kharidar__username__contains=username) |
                Q(gholname_kharid__foro0shande__username__contains=username)
            )

        paginator = Paginator(kharid_foro0shs, NUM_OF_PAGINATION_ITEMS)
        page = request.query_params.get("page")

        try:
            kharid_foro0shs = paginator.page(page)
        except PageNotAnInteger:
            kharid_foro0shs = paginator.page(1)
        except EmptyPage:
            kharid_foro0shs = []

        serializers = self.serializer_class(kharid_foro0shs, many=True).data
        return response({"baygani": serializers, "num_of_pages": paginator.num_pages})

    def post(self, request):
        """
        ijAd e yek sanad e bAygAni
        --------
        parameters

            - name: id
            type: integer
            description: id e baygani
            optional: false

            - name: current_state
            type: integer
            description: vaziat e kono0ni
            optional: false

            - name: vekalat_kari_engheza
            type: timestamp
            description: enghezA vekalat kAri
            optional: true

            - name: sanad_dar_rahn_engheza
            type: timestamp
            description: enghezA e sanad dar rahn
            optional: true

            - name: vekalat_foro0sh_engheza
            type: timestamp
            description: enghezA e vekalat foro0sh
            optional: true

            - name: sanad_sabz
            type: bool
            description: sanad sabz dAre ya na
            optional: true

            - name: sanad_company
            type: bool
            description: sanad company dAre ya na
            optional: true

            - name: sanad_ghati
            type: bool
            description: sanad_ghati dAre ya na
            optional: true

            - name: taiidie_pelak
            type: bool
            description: taiidie pelAk dAre ya na
            optional: true

            - name: vekalat_kari
            type: bool
            description: vekalat kAri dAre ya na
            optional: true

            - name: vekalat_foro0sh
            type: bool
            description: vekalat foro0sh dAre ya na
            optional: true

            - name: avArez_ta
            type: timestamp
            description: avArez tA sAl e felAn
            optional: false

            - name: enter_datetime
            type: timestamp
            description: tArikh e voro0d
            optional: false

            - name: tahvil_dahande_id
            type: integer
            description: id e user e tahvil dahande

            - name: description
            type: string
            description: tozihAt !
            optional: true

            - name: vakil_id
            type: integer
            description: id e user e vakil

        Response Messages
            - code: 200
            response: {
                "status": "ok"
            }

        Error Messages
            - code: 002
            response: {"status":"object not found","code":"002"}

            - code: 001
            response: {"status":"not valid parameters","code":"001"}
        """
        parser = RequestParser(request.data)
        try:
            baygani_id = parser.int("id", None, is_optional=False)
            current_state = parser.int("current_state", None)
            vekalat_kari_engheza = parser.timestamp("vekalat_kari_engheza", None)
            sanad_dar_rahn_engheza = parser.timestamp("sanad_dar_rahn_engheza", None)
            vekalat_foro0sh_engheza = parser.timestamp("vekalat_foro0sh_engheza", None)
            sanad_sabz = parser.bool("sanad_sabz", False)
            sanad_company = parser.bool("sanad_company", False)
            sanad_ghati = parser.bool("sanad_ghati", False)
            taiidie_pelak = parser.bool("taiidie_pelak", False)
            vekalat_kari = parser.bool("vekalat_kari", False)
            vekalat_foro0sh = parser.bool("vekalat_foro0sh", False)
            avArez_ta = parser.timestamp("avArez_ta", None)
            enter_datetime = parser.timestamp("enter_datetime", None, is_optional=False)
            tahvil_dahande_id = parser.int("tahvil_dahande_id", None, is_optional=True)
            vakil_id = parser.int("vakil_id", None, is_optional=True)
            description = parser.str("description", "")
        except EmamParameterError as e:
            return raise_error("PARAMETERS_ERROR", e.field_name)

        try:
            kharid_foro0sh = KharidForo0shCar.objects.get(id=baygani_id)
        except KharidForo0shCar.DoesNotExist:
            return raise_error("NOT_FOUND_ERROR", "BAYGANI")

        if tahvil_dahande_id is not None:
            try:
                tahvil_dahande = User.objects.get(id=tahvil_dahande_id)
            except User.DoesNotExist:
                return raise_error("NOT_FOUND_ERROR", "Tahvil girande")
        else:
            tahvil_dahande = None

        if vakil_id is not None:
            try:
                vakil = User.objects.get(id=vakil_id)
            except User.DoesNotExist:
                return raise_error("NOT_FOUND_ERROR", "Vakil")
        else:
            vakil = None

        numbers = ShomareBaygani.objects.filter(remained__gt=0).order_by("created_at")
        if numbers:
            shomare_baygani = numbers.first()
            shomare_baygani.remained -= 1
            shomare_baygani.save()
        else:
            numbers = ShomareBaygani.objects.filter(remained=0).order_by("created_at")
            if numbers:
                shomare_baygani = numbers.last()
                number = shomare_baygani.number + 1
                shomare_baygani = ShomareBaygani.objects.create(number=number, remained=1)
            else:
                shomare_baygani = ShomareBaygani.objects.create(number=1, remained=1)

        kharid_foro0sh.update(
            shomare_baygani=shomare_baygani,
            current_state=current_state,
            vekalat_kari_engheza=vekalat_kari_engheza,
            sanad_dar_rahn_engheza=sanad_dar_rahn_engheza,
            vekalat_foro0sh_engheza=vekalat_foro0sh_engheza,
            sanad_sabz=sanad_sabz,
            sanad_company=sanad_company,
            sanad_ghati=sanad_ghati,
            taiidie_pelak=taiidie_pelak,
            vekalat_kari=vekalat_kari,
            vekalat_foro0sh=vekalat_foro0sh,
            avArez_ta=avArez_ta,
            enter_datetime=enter_datetime,
            tahvil_dahande=tahvil_dahande,
            description=description,
            is_baygani=True,
            vakil=vakil
        )

        return response({"status": "ok"})


class BayganiView(APIView):
    serializer_class = BayganiListSerializer
    permission_classes = [IsAuthenticated, IsICarAdmin]
    authentication_classes = [TokenAuthentication]

    def get(self, request, **kwargs):
        """
        daryAfte yek mored az baygani
        --------
        parameters
            - name: page
            type: int
            description: shomAre pagination -> ?page=1

            - name: id
            type: int
            description: id e object e morede nazar

        Response Messages
            - code: 200
            response: {
                "id": 50,
                "shomare_baygani": 12,
                "sanad_dar_rahn_engheza": "1531231531",
                "foro0shande": "خلیل رزاقیان",
                "kharid_namayeshgah_datetime": "1531231531",
                "kharidar": "حسن قاضی خانی",
                "gholname_datetime": "1531231564",
                "current_state": 41,
                "madarek_tahvili": {
                    "id": 8,
                    "avArez_ta": null,
                    "sanad_sabz": true,
                    "sanad_company": true,
                    "sanad_ghati": true,
                    "taiidie_pelak": false,
                    "vekalat_kari": false,
                    "vekalat_foro0sh": false
                },
                "pelak_ghadim": "11ق11111",
                "pelak_jadid": "",
                "car": {
                    "name": "سواری",
                    "model": "1200",
                    "shasi": "11111",
                    "color": "سبز",
                    "system": "پراید"
                },
                "enter_datetime" : 1531231531
                "saheb_pelak_ghadim": "خلیل رزاقیان",
                "dar_hal_hazer": 0,
                "vakil": {"id", "first_name", "last_name", "code_melli", "mobile_number", "father", "shomare_shenasname",
                  "address", "sadere", "phone_number", "code_posti"}
                "description": "",
                "vekalat_kari_engheza": null,
                "vekalat_foro0sh_engheza": null,
                "tahvil_dahande": {"id", "first_name", "last_name", "code_melli", "mobile_number", "father", "shomare_shenasname",
                  "address", "sadere", "phone_number", "code_posti"}
            }
        """
        parser = RequestParser(kwargs)
        try:
            baygani_id = parser.int("id", None, is_optional=False)
        except EmamParameterError as e:
            return raise_error("PARAMETERS_ERROR", e.field_name)

        try:
            kharid_foro0sh = KharidForo0shCar.objects.get(id=baygani_id)
        except KharidForo0shCar.DoesNotExist:
            return raise_error("NOT_FOUND_ERROR", "BAYGANI")

        serializer = ShowBayganiSerializer(kharid_foro0sh).data
        return response(serializer)

    def put(self, request, **kwargs):
        """
        virAyesh e sanad e bAygAni
        --------
        parameters
            - name: id
            type: int
            description: id e object e morede nazar

            - name: current_state
            type: int
            description: vaziate kono0ni | 0: vekAlate kAri nadArad, 1: vekAlate kAri dArad, 2: sanad dar rahn ast, 3: vekAlate foro0sh dArad, 4: taviz e pelak shod, 5:sanad e ghati khord
            optional: True

            - name: vakil_id
            type: int
            description: id e user e vakil
            optional: True

            -name: vekalat_kari_engheza
            type: timestamp

            -name: vekalat_foro0sh_engheza
            type: timestamp

            -name: sanad_dar_rahn_engheza
            type: timestamp

            -name: description
            type: timestamp

        Response Messages
            - code: 200
            response: {
                "status": True
            }

        Error Messages
            - code: 002
            response: {"status":"object not found","code":"002"}

            - code: 001
            response: {"status":"not valid parameters","code":"001"}
        """
        parser = RequestParser(kwargs)
        data_parser = RequestParser(request.data)
        try:
            baygani_id = parser.int("id", None, is_optional=False)
            current_state = data_parser.int("current_state", None)
            vakil_id = data_parser.int("vakil_id", None)
            vekalat_kari_engheza = data_parser.timestamp("vekalat_kari_engheza", None, is_optional=True)
            vekalat_foro0sh_engheza = data_parser.timestamp("vekalat_foro0sh_engheza", None, is_optional=True)
            sanad_dar_rahn_engheza = data_parser.timestamp("sanad_dar_rahn_engheza", None, is_optional=True)
            description = data_parser.str("description", None, is_optional=True)
        except EmamParameterError as e:
            return raise_error("PARAMETERS_ERROR", e.field_name)

        try:
            kharid_foro0sh = KharidForo0shCar.objects.get(id=baygani_id)
        except KharidForo0shCar.DoesNotExist:
            return raise_error("NOT_FOUND_ERROR", "BAYGANI")

        if current_state is not None:
            kharid_foro0sh.current_state = current_state

        if vekalat_kari_engheza is not None:
            kharid_foro0sh.vekalat_kari_engheza = vekalat_kari_engheza
        if vekalat_foro0sh_engheza is not None:
            kharid_foro0sh.vekalat_foro0sh_engheza = vekalat_foro0sh_engheza
        if sanad_dar_rahn_engheza is not None:
            kharid_foro0sh.sanad_dar_rahn_engheza = sanad_dar_rahn_engheza

        if vakil_id:
            try:
                vakil = User.objects.get(id=vakil_id)
            except User.DoesNotExist:
                return raise_error("PARAMETERS_ERROR", "USER")

            kharid_foro0sh.vakil = vakil

        if description is not None:
            kharid_foro0sh.description = description

        if current_state is not None or vakil_id or vekalat_kari_engheza is not None or vekalat_foro0sh_engheza is not None or description is not None or sanad_dar_rahn_engheza is not None:
            kharid_foro0sh.save()

        return response({"status": "ok"})


class MotefarregheView(APIView):
    permission_classes = (IsAuthenticated, IsICarAdmin)
    authentication_classes = (TokenAuthentication,)

    def get(self, request):
        """
        daryAfte list e mavArede mojo0d dar baygani ba filtration
        --------
        parameters
            - name: page
            type: int
            description: shomAre pagination -> ?page=1

            - name: subject
            type: string
            description: onvAn e sanad

            - name: shomare_baygani
            type: int
            description: shomAre bAygAni

            - name: type
            type: string
            description: type e sanad : car_sanads, typical
            required: True !

        Response Messages
            - code: 200
            response:{
                "num_of_pages": 1,
                "sanads": [
                    {
                        "subject": "سند ملک",
                        "enter_datetime": "1527297824",
                        "description": "خوب"
                    }
                ]
            }
        """
        parser = RequestParser(request.query_params)
        try:
            subject = parser.str("subject", None)
            shomare_baygani = parser.int("shomare_baygani", None)
            sanad_type = parser.str("type", None, is_optional=False)
        except EmamParameterError as e:
            return raise_error("PARAMETERS_ERROR", e.field_name)

        filter_list = {}

        if subject:
            filter_list["subject__contains"] = subject

        if shomare_baygani:
            filter_list["shomare_baygani__number"] = shomare_baygani

        if sanad_type == "car_sanads":
            sanads = SanadMotefarregheCar.objects.filter(**filter_list).order_by("-created_at")
            serializer_class = SanadMotefarregheCarListSerializer
        elif sanad_type == "typical":
            sanads = SanadMotefarreghe.objects.filter(**filter_list).order_by("-created_at")
            serializer_class = SanadMotefarregheListSerializer
        else:
            return response([])

        paginator = Paginator(sanads, NUM_OF_PAGINATION_ITEMS)
        page = request.query_params.get("page")

        try:
            sanads = paginator.page(page)
        except PageNotAnInteger:
            sanads = paginator.page(1)
        except EmptyPage:
            sanads = []

        serializers = serializer_class(sanads, many=True).data
        return response({"sanads": serializers, "num_of_pages": paginator.num_pages})

    def post(self, request):
        """
        ijAd e yek sanad e motefarreghe
        --------
        parameters
            - name: type
            type: string
            description: type e sanad : car_sanad, typical
            required: True !

            agar khodro bo0d:
                - name: subject
                type: string
                description: nAm e khodro

                -name: system
                type: string

                -name: tip
                type: string

                -name: model
                type: string

                -name: color
                type: string

                -name: pelak_ghadim
                type: string

                -name: pelak_jadid
                type: string

                -name: shasi
                type: string

                -name: description
                type: string

                -name: enter_datetime
                type: timestamp
                description: tArikh e voro0d

                -name: sanad_sabz
                type: bool

                -name: sanad_company
                type: bool

                -name: sanad_ghati
                type: bool

                -name: taiidie_pelak
                type: bool

                -name: vekalat_kari
                type: bool

                -name: vekalat_foro0sh
                type: bool

                -name: avArez_ta
                type: timestamp

            agar typical bo0d:
                - name: subject
                type: string
                description: onvAn e sanad

                -name: description
                type: string


        Response Messages
            - code: 200
            response:{
                "status": "ok"
            }
        """
        numbers = ShomareBaygani.objects.filter(remained__gt=0).order_by("created_at")
        if numbers:
            shomare_baygani = numbers.first()
            shomare_baygani.remained -= 1
            shomare_baygani.save()
        else:
            numbers = ShomareBaygani.objects.filter(remained=0).order_by("created_at")
            if numbers:
                shomare_baygani = numbers.last()
                number = shomare_baygani.number + 1
                shomare_baygani = ShomareBaygani.objects.create(number=number, remained=1)
            else:
                shomare_baygani = ShomareBaygani.objects.create(number=1, remained=1)

        parser = RequestParser(request.data)
        try:
            sanad_type = parser.str("type", None, is_optional=False)
        except EmamParameterError as e:
            return raise_error("PARAMETERS_ERROR", e.field_name)

        if sanad_type == "car_sanad":
            try:
                subject = parser.str("subject", None, is_optional=False)
                system = parser.str("system", None, is_optional=False)
                tip = parser.str("tip", None, is_optional=False)
                model = parser.str("model", None, is_optional=False)
                color = parser.str("color", None, is_optional=False)
                pelak_ghadim = parser.str("pelak_ghadim", None, is_optional=True)
                pelak_jadid = parser.str("pelak_jadid", None, is_optional=True)
                shasi = parser.str("shasi", None, is_optional=False)
                description = parser.str("description", None, is_optional=True)
                enter_datetime = parser.timestamp("enter_datetime", None, is_optional=False)
                sanad_sabz = parser.bool("sanad_sabz", None, is_optional=True)
                sanad_company = parser.bool("sanad_company", None, is_optional=True)
                sanad_ghati = parser.bool("sanad_ghati", None, is_optional=True)
                taiidie_pelak = parser.bool("taiidie_pelak", None, is_optional=True)
                vekalat_kari = parser.bool("vekalat_kari", None, is_optional=True)
                vekalat_foro0sh = parser.bool("vekalat_foro0sh", None, is_optional=True)
                avArez_ta = parser.timestamp("avArez_ta", None, is_optional=True)
            except EmamParameterError as e:
                return raise_error("PARAMETERS_ERROR", e.field_name)

            madarek_tahvili = MadarekTahvili.objects.create(
                sanad_sabz=sanad_sabz,
                sanad_company=sanad_company,
                sanad_ghati=sanad_ghati,
                taiidie_pelak=taiidie_pelak,
                vekalat_kari=vekalat_kari,
                vekalat_foro0sh=vekalat_foro0sh,
                avArez_ta=avArez_ta
            )

            SanadMotefarregheCar.objects.create(
                shomare_baygani=shomare_baygani,
                subject=subject,
                system=system,
                tip=tip,
                model=model,
                color=color,
                pelak_ghadim=pelak_ghadim,
                pelak_jadid=pelak_jadid,
                shasi=shasi,
                description=description,
                enter_datetime=enter_datetime,
                madarek_tahvili=madarek_tahvili
            )

        elif sanad_type == "typical":
            try:
                subject = parser.str("subject", None, is_optional=False)
                description = parser.str("description", None, is_optional=False)
            except EmamParameterError as e:
                return raise_error("PARAMETERS_ERROR", e.field_name)

            SanadMotefarreghe.objects.create(
                shomare_baygani=shomare_baygani,
                subject=subject,
                description=description
            )
        else:
            return raise_error("PARAMETERS_ERROR", "type")

        return response({"status": "ok"})


class SanadMotefarregheView(APIView):
    def get(self, request, **kwargs):
        sanad_id = kwargs["id"]

        try:
            sanad = SanadMotefarreghe.objects.get(id=sanad_id)
        except SanadMotefarreghe.DoesNotExist:
            return raise_error("NOT_FOUND_ERROR", "Sanad")

        serializer = SanadMotefarregheListSerializer(sanad).data
        return response(serializer)

    def put(self, request, **kwargs):
        sanad_id = kwargs["id"]

        try:
            subject = request.data.get("subject", None)
            description = request.data.get("description", None)
        except KeyError:
            return raise_error("PARAMETERS_ERROR")

        try:
            sanad = SanadMotefarreghe.objects.get(id=sanad_id)
        except SanadMotefarreghe.DoesNotExist:
            return raise_error("NOT_FOUND_ERROR", "Sanad")

        sanad.subject = subject
        sanad.description = description
        sanad.save()

        return response({"status": "ok"})

    def delete(self, request, **kwargs):
        sanad_id = kwargs["id"]

        try:
            sanad = SanadMotefarreghe.objects.get(id=sanad_id)
        except SanadMotefarreghe.DoesNotExist:
            return raise_error("NOT_FOUND_ERROR", "Sanad")

        sanad.delete()

        return response({"status": "ok"})


class SanadMotefarregheMashinView(APIView):
    def get(self, request, **kwargs):
        sanad_id = kwargs["id"]

        try:
            sanad = SanadMotefarregheCar.objects.get(id=sanad_id)
        except SanadMotefarregheCar.DoesNotExist:
            return raise_error("NOT_FOUND_ERROR", "Sanad")

        serializer = SanadMotefarregheCarSerializer(sanad).data
        return response(serializer)

    def put(self, request, **kwargs):
        sanad_id = kwargs["id"]
        parser = RequestParser(request.data)

        try:
            subject = parser.str("subject", None, is_optional=True)
            system = parser.str("system", None, is_optional=True)
            tip = parser.str("tip", None, is_optional=True)
            model = parser.str("model", None, is_optional=True)
            color = parser.str("color", None, is_optional=True)
            pelak_ghadim = parser.str("pelak_ghadim", None, is_optional=True)
            pelak_jadid = parser.str("pelak_jadid", None, is_optional=True)
            shasi = parser.str("shasi", None, is_optional=True)
            description = parser.str("description", None, is_optional=True)
            sanad_sabz = parser.bool("sanad_sabz", None, is_optional=True)
            sanad_company = parser.bool("sanad_company", None, is_optional=True)
            sanad_ghati = parser.bool("sanad_ghati", None, is_optional=True)
            taiidie_pelak = parser.bool("taiidie_pelak", None, is_optional=True)
            vekalat_kari = parser.bool("vekalat_kari", None, is_optional=True)
            vekalat_foro0sh = parser.bool("vekalat_foro0sh", None, is_optional=True)
            avArez_ta = parser.timestamp("avArez_ta", None, is_optional=True)
        except EmamParameterError as e:
            return raise_error("PARAMETERS_ERROR", e.field_name)

        try:
            sanad = SanadMotefarregheCar.objects.get(id=sanad_id)
        except SanadMotefarregheCar.DoesNotExist:
            return raise_error("NOT_FOUND_ERROR", "Sanad")

        sanad.update(subject, system, tip, model, color, pelak_ghadim, pelak_jadid, shasi, sanad_sabz, sanad_company,
                     sanad_ghati, taiidie_pelak, vekalat_kari, vekalat_foro0sh, avArez_ta, description)
        sanad.save()

        return response({"status": "ok"})

    def delete(self, request, **kwargs):
        sanad_id = kwargs["id"]

        try:
            sanad = SanadMotefarregheCar.objects.get(id=sanad_id)
        except SanadMotefarregheCar.DoesNotExist:
            return raise_error("NOT_FOUND_ERROR", "Sanad")

        sanad.delete()

        return response({"status": "ok"})


class ExitView(APIView):
    serializer_class = BayganiListSerializer
    permission_classes = [IsAuthenticated, IsICarAdmin]
    authentication_classes = [TokenAuthentication]

    def post(self, request, **kwargs):
        """
        khoro0j e yek sanad az bAygAni
        --------
        parameters

            - name: id
            type: integer
            description: id e baygani
            optional: false

            - name: sanad_sabz
            type: bool
            description: sanad sabz dAre ya na
            optional: true

            - name: sanad_company
            type: bool
            description: sanad company dAre ya na
            optional: true

            - name: sanad_ghati
            type: bool
            description: sanad_ghati dAre ya na
            optional: true

            - name: taiidie_pelak
            type: bool
            description: taiidie pelAk dAre ya na
            optional: true

            - name: vekalat_kari
            type: bool
            description: vekalat kAri dAre ya na
            optional: true

            - name: vekalat_foro0sh
            type: bool
            description: vekalat foro0sh dAre ya na
            optional: true

            - name: avArez_ta
            type: timestamp
            description: avArez tA sAl e felAn
            optional: true

            - name: tahvil_girande_id
            type: integer
            description: id e user e tahvil dahande
            optional: false

            - name: description
            type: string
            description: tozihAt !
            optional: true

            - name: bargasht_ehtemali_datetime
            type: timestamp
            description: tArikh e bargasht e ehtemAli e sanad
            optional: false

            - name: need_to_reminder
            type: integer
            description: niAz be yAd Avari | 0: bedo0n e niAz be yAd Avari, 1: 1 ro0z baad, 2: 3 ro0z baad, 3: har 2 ro0z yek bAr
            optional: true

            - name: ellat
            type: integer
            description: ellate khoro0j e sanad | 0: mahzar, 1: AzAd sAzi, 2: tahvil, 3: tahvil e kharidAr
            optional: false

            - name: datetime
            type: timestamp
            description: tArikh e khoro0j
            optional: false

        Response Messages
            - code: 200
            response: {
                "status": "ok"
            }

        Error Messages
            - code: 002
            response: {"status":"object not found","code":"002"}

            - code: 001
            response: {"status":"not valid parameters","code":"001"}

            - code: 005
            response: {"status": "You can do this just one time", "code": "005"}
        """
        parser = RequestParser(request.data)
        try:
            baygani_id = kwargs["id"]
            datetime = parser.timestamp("datetime", None, is_optional=False)
            tahvil_girande_id = parser.int("tahvil_girande_id", None, is_optional=False)
            ellat = parser.int("ellat", None, is_optional=False)
            sanad_sabz = parser.bool("sanad_sabz", False, is_optional=True)
            sanad_company = parser.bool("sanad_company", False, is_optional=True)
            sanad_ghati = parser.bool("sanad_ghati", False, is_optional=True)
            taiidie_pelak = parser.bool("taiidie_pelak", False, is_optional=True)
            vekalat_kari = parser.bool("vekalat_kari", False, is_optional=True)
            vekalat_foro0sh = parser.bool("vekalat_foro0sh", False, is_optional=True)
            avArez_ta = parser.timestamp("avArez_ta", None, is_optional=True)
            bargasht_datetime_ehtemali = parser.timestamp("bargasht_ehtemali_datetime", None, is_optional=False)
            need_to_reminder = parser.int("need_to_reminder", 0)
            description = parser.str("description", "", is_optional=True)
        except EmamParameterError as e:
            return raise_error("PARAMETERS_ERROR", e.field_name + "|url: baygani/exit/")

        try:
            kharid_foro0sh = KharidForo0shCar.objects.get(id=baygani_id)
        except KharidForo0shCar.DoesNotExist:
            return raise_error("NOT_FOUND_ERROR", "BAYGANI")

        try:
            tahvil_girande = User.objects.get(id=tahvil_girande_id)
        except User.DoesNotExist:
            return raise_error("NOT_FOUND_ERROR", "TAHVIL_GIRANDE")

        madarek_tahvili = MadarekTahvili.objects.create(
            sanad_sabz=sanad_sabz,
            sanad_company=sanad_company,
            sanad_ghati=sanad_ghati,
            taiidie_pelak=taiidie_pelak,
            vekalat_kari=vekalat_kari,
            vekalat_foro0sh=vekalat_foro0sh,
            avArez_ta=avArez_ta
        )

        Exit.objects.create(
            kharid_foro0sh=kharid_foro0sh,
            datetime=datetime,
            tahvil_girande=tahvil_girande,
            ellat=ellat,
            madarek_tahvili=madarek_tahvili,
            bargasht_datetime_ehtemali=bargasht_datetime_ehtemali,
            need_to_reminder=need_to_reminder,
        )

        if sanad_sabz:
            try:
                kharid_foro0sh.madarek_tahvili.sanad_sabz = False
            except AttributeError:
                pass
        if sanad_company:
            try:
                kharid_foro0sh.madarek_tahvili.sanad_company = False
            except AttributeError:
                pass
        if sanad_ghati:
            try:
                kharid_foro0sh.madarek_tahvili.sanad_ghati = False
            except AttributeError:
                pass
        if taiidie_pelak:
            try:
                kharid_foro0sh.madarek_tahvili.taiidie_pelak = False
            except AttributeError:
                pass
        if vekalat_kari:
            try:
                kharid_foro0sh.madarek_tahvili.vekalat_kari = False
            except AttributeError:
                pass
        if vekalat_foro0sh:
            try:
                kharid_foro0sh.madarek_tahvili.vekalat_foro0sh = False
            except AttributeError:
                pass
        if avArez_ta:
            try:
                kharid_foro0sh.madarek_tahvili.avArez_ta = None
            except AttributeError:
                pass

        kharid_foro0sh.madarek_tahvili.save()
        kharid_foro0sh.dar_hal_hazer = ellat
        kharid_foro0sh.is_exited = True
        kharid_foro0sh.description += "\n"
        kharid_foro0sh.description += description
        if ellat == 2:
            kharid_foro0sh.is_baygani = False
            kharid_foro0sh.shomare_baygani.remained += 1
            kharid_foro0sh.shomare_baygani.save()
        kharid_foro0sh.save()

        return response({"status": "ok"})


class EnterView(APIView):
    serializer_class = BayganiListSerializer
    permission_classes = [IsAuthenticated, IsICarAdmin]
    authentication_classes = [TokenAuthentication]

    def post(self, request, **kwargs):
        """
        voro0d e yek sanad be bAygAni
        --------
        parameters

            - name: id
            type: integer
            description: id e baygani
            optional: false

            - name: sanad_sabz
            type: bool
            description: sanad sabz dAre ya na
            optional: true

            - name: sanad_company
            type: bool
            description: sanad company dAre ya na
            optional: true

            - name: sanad_ghati
            type: bool
            description: sanad_ghati dAre ya na
            optional: true

            - name: taiidie_pelak
            type: bool
            description: taiidie pelAk dAre ya na
            optional: true

            - name: vekalat_kari
            type: bool
            description: vekalat kAri dAre ya na
            optional: true

            - name: vekalat_foro0sh
            type: bool
            description: vekalat foro0sh dAre ya na
            optional: true

            - name: avArez_ta
            type: timestamp
            description: avArez tA sAl e felAn
            optional: true

            - name: tahvil_dahande_id
            type: integer
            description: id e user e tahvil dahande
            optional: false

            - name: description
            type: string
            description: tozihAt !
            optional: true

            - name: is_fak_rahn
            type: bool
            description: inke sanad fak e rahn shod ya na
            optional: true

            - name: datetime
            type: timestamp
            description: tArikh e voro0d
            optional: false

        Response Messages
            - code: 200
            response: {
                "status": "ok"
            }

        Error Messages
            - code: 002
            response: {"status":"object not found","code":"002"}

            - code: 001
            response: {"status":"not valid parameters","code":"001"}

            - code: 005
            response: {"status": "You can do this just one time", "code": "005"}
        """
        parser = RequestParser(request.data)
        try:
            baygani_id = kwargs["id"]
            datetime = parser.timestamp("datetime", None, is_optional=False)
            tahvil_dahande_id = parser.int("tahvil_dahande_id", None, is_optional=False)
            sanad_sabz = parser.bool("sanad_sabz", False, is_optional=True)
            sanad_company = parser.bool("sanad_company", False, is_optional=True)
            sanad_ghati = parser.bool("sanad_ghati", False, is_optional=True)
            taiidie_pelak = parser.bool("taiidie_pelak", False, is_optional=True)
            vekalat_kari = parser.bool("vekalat_kari", False, is_optional=True)
            vekalat_foro0sh = parser.bool("vekalat_foro0sh", False, is_optional=True)
            avArez_ta = parser.timestamp("avArez_ta", None, is_optional=True)
            is_fak_rahn = parser.bool("is_fak_rahn", False, is_optional=True)
            description = parser.str("description", "", is_optional=True)
        except EmamParameterError as e:
            return raise_error("PARAMETERS_ERROR", e.field_name)

        try:
            kharid_foro0sh = KharidForo0shCar.objects.get(id=baygani_id)
        except KharidForo0shCar.DoesNotExist:
            return raise_error("NOT_FOUND_ERROR", "BAYGANI")

        try:
            tahvil_dahande = User.objects.get(id=tahvil_dahande_id)
        except User.DoesNotExist:
            return raise_error("NOT_FOUND_ERROR", "TAHVIL_DAHANDE")

        madarek_tahvili = MadarekTahvili.objects.create(
            sanad_sabz=sanad_sabz,
            sanad_company=sanad_company,
            sanad_ghati=sanad_ghati,
            taiidie_pelak=taiidie_pelak,
            vekalat_kari=vekalat_kari,
            vekalat_foro0sh=vekalat_foro0sh,
            avArez_ta=avArez_ta
        )

        Enter.objects.create(
            kharid_foro0sh=kharid_foro0sh,
            datetime=datetime,
            madarek_tahvili=madarek_tahvili,
            tahvil_dahande=tahvil_dahande,
            is_fak_rahn=is_fak_rahn
        )

        if is_fak_rahn:
            kharid_foro0sh.current_state = 1

        if sanad_sabz:
            try:
                kharid_foro0sh.madarek_tahvili.sanad_sabz = True
            except AttributeError:
                pass
        if sanad_company:
            try:
                kharid_foro0sh.madarek_tahvili.sanad_company = True
            except AttributeError:
                pass
        if sanad_ghati:
            try:
                kharid_foro0sh.madarek_tahvili.sanad_ghati = True
            except AttributeError:
                pass
        if taiidie_pelak:
            try:
                kharid_foro0sh.madarek_tahvili.taiidie_pelak = True
            except AttributeError:
                pass
        if vekalat_kari:
            try:
                kharid_foro0sh.madarek_tahvili.vekalat_kari = True
            except AttributeError:
                pass
        if vekalat_foro0sh:
            try:
                kharid_foro0sh.madarek_tahvili.vekalat_foro0sh = True
            except AttributeError:
                pass
        if avArez_ta:
            try:
                kharid_foro0sh.madarek_tahvili.avArez_ta = avArez_ta
            except AttributeError:
                pass

        kharid_foro0sh.madarek_tahvili.save()
        kharid_foro0sh.is_exited = False
        kharid_foro0sh.description += "\n"
        kharid_foro0sh.description += description
        kharid_foro0sh.save()

        return response({"status": "ok"})


class KharidForo0shListView(APIView):
    serializer_class = BayganiListSerializer
    permission_classes = (IsAuthenticated, IsICarAdmin)
    authentication_classes = (TokenAuthentication,)

    def get(self, request):
        """
        daryAfte list e mavArede mojo0d dar baygani ba filtration
        --------
        parameters
            - name: page
            type: int
            description: shomAre pagination -> ?page=1

            - name: name
            type: string
            description: nAm e kharidAr ya foro0shande

            - name: code_melli
            type: string
            description: code melli e kharidAr ya foro0shande

            - name: shasi
            type: string
            description: shomare shasi e mashin

            - name: pelak_ghadim
            type: string
            description: pelak e ghadim e mashin

            - name: pelak_jadid
            type: string
            description: pelak e jadid e mashin

            - name: saheb_pelak_ghadim_id
            type: int
            description: id e saheb pelak e ghadim

            - name: vakil_id
            type: int
            description: id e vakil

            - name: khodro
            type: string
            description: nAme khodro

            - name: system
            type: string
            description: system e mashin

            - name: etmam_vekalat_foro0sh
            type: timestamp
            description: vAzehe dg

            - name: etmam_vekalat_kari
            type: timestamp
            description: inam vAzehe

        Response Messages
            - code: 200
            response: {
                "num_of_pages": 1,
                "baygani": [
                    {
                        "id": 1,
                        "foro0shande": "یبلاتن بلاتن",
                        "kharid_namayeshgah_datetime": "1525677305",
                        "kharidar": "نتاز هتاذ",
                        "gholname_datetime": "1526446920",
                        "current_state": 2,
                        "vekalat_kari_engheza": "1527130993",
                        "sanad_dar_rahn_engheza": "1527208631",
                        "sanad_dar_rahn_vekalat_kari": false,
                        "vekalat_foro0sh_engheza": "1527208634",
                        "taviz_pelak_shode_state": 0,
                        "madarek_tahvili": {
                            "id": 2,
                            "avArez_ta": null,
                            "sanad_sabz": false,
                            "sanad_company": true,
                            "sanad_ghati": false,
                            "taiidie_pelak": false,
                            "vekalat_kari": false,
                            "vekalat_foro0sh": false
                        },
                        "tahvil_dahande": "حسین کریمی",
                        "description": "بسیار زیبا",
                        "pelak_ghadim": "asd",
                        "pelak_jadid": "asd",
                        "car": {
                            "name": "پورشه",
                            "model": "بالا",
                            "shasi": "2345",
                            "system": "kado0tanbal"
                        },
                        "saheb_pelak_ghadim": "یبلاتن بلاتن"
                    }
                ]
            }
        """
        parser = RequestParser(request.query_params)
        try:
            name = parser.str("name", None)
            shasi = parser.str("shasi", None)
            username = parser.str("code_melli", None)
            pelak_ghadim = parser.str("pelak_ghadim", None)
            pelak_jadid = parser.str("pelak_jadid", None)
            etmam_vekalat_kari = parser.timestamp("etmam_vekalat_kari", None)
            etmam_vekalat_foro0sh = parser.timestamp("etmam_vekalat_foro0sh", None)
            car_name = parser.str("khodro", None)
            system = parser.str("system", None)
            vakil_id = parser.int("vakil_id", None)
            saheb_pelak_ghadim_id = parser.int("saheb_pelak_ghadim_id", None)
        except EmamParameterError as e:
            return raise_error("PARAMETERS_ERROR", e.field_name)

        filter_list = {"gholname_kharid__faskh__isnull": True, "gholname_foro0sh__faskh__isnull": True,
                       "is_baygani": False}

        if saheb_pelak_ghadim_id:
            filter_list["gholname_kharid__foro0shande__id"] = saheb_pelak_ghadim_id

        if pelak_ghadim:
            filter_list["pelak_ghadim__contains"] = pelak_ghadim

        if vakil_id:
            filter_list["vakil__id"] = vakil_id

        if pelak_jadid:
            filter_list["pelak_jadid__contains"] = pelak_ghadim

        if car_name:
            filter_list["gholname_kharid__car__name__contains"] = car_name

        if system:
            filter_list["gholname_kharid__car__system__contains"] = system

        if etmam_vekalat_foro0sh:
            filter_list["vekalat_foro0sh_engheza__lte"] = etmam_vekalat_foro0sh

        if etmam_vekalat_kari:
            filter_list["vekalat_kari_engheza__lte"] = etmam_vekalat_kari

        if shasi:
            filter_list["gholname_kharid__car__shasi__contains"] = shasi

        kharid_foro0shs = KharidForo0shCar.objects.filter(**filter_list).order_by("-created_at")

        if name:
            kharid_foro0shs = kharid_foro0shs.filter(
                Q(gholname_foro0sh__kharidar__first_name__contains=name) |
                Q(gholname_foro0sh__kharidar__last_name__contains=name) |
                Q(gholname_kharid__foro0shande__first_name__contains=name) |
                Q(gholname_kharid__foro0shande__last_name__contains=name)
            )

        if username:
            kharid_foro0shs = kharid_foro0shs.filter(
                Q(gholname_foro0sh__kharidar__username__contains=username) |
                Q(gholname_kharid__foro0shande__username__contains=username)
            )

        paginator = Paginator(kharid_foro0shs, NUM_OF_PAGINATION_ITEMS)
        page = request.query_params.get("page")

        try:
            kharid_foro0shs = paginator.page(page)
        except PageNotAnInteger:
            kharid_foro0shs = paginator.page(1)
        except EmptyPage:
            kharid_foro0shs = []

        serializers = self.serializer_class(kharid_foro0shs, many=True).data
        return response({"baygani": serializers, "num_of_pages": paginator.num_pages})


class ShomareBayganiView(APIView):
    permission_classes = (IsAuthenticated, IsICarAdmin)
    authentication_classes = (TokenAuthentication,)

    def get(self, request, *args, **kwargs):
        """
        daryAfte shomAre baygani be so0rate automatic
        --------

        Response Messages
            - code: 200
            response: {
                "shomare_baygani": 10012
            }

        """
        numbers = ShomareBaygani.objects.filter(remained__gt=0).order_by("created_at")
        if numbers:
            shomare_baygani = numbers.first()
            number = shomare_baygani.number
        else:
            numbers = ShomareBaygani.objects.filter(remained=0).order_by("created_at")
            if numbers:
                shomare_baygani = numbers.last()
                number = shomare_baygani.number + 1
            else:
                number = 1
        return response({"shomare_baygani": number})
