# -*- coding: utf-8 -*-
from __future__ import unicode_literals

from django.core.paginator import Paginator, PageNotAnInteger, EmptyPage
from django.db.models import Q
from django.db.models.aggregates import Sum
from django.db.models.expressions import F
from django.utils import timezone
from rest_framework.authentication import TokenAuthentication
from rest_framework.permissions import IsAuthenticated
from rest_framework.views import APIView

from applications.accounting.models import (
    DaryaftAzTarafHesab, DaftarKol, Chek, Bank, PardakhtBeTarafHesab,
    TalabBedeh,
    HesabDaftarKol,
    TalabBedehGhest,
    BankTransfer,
    ChekStatusDelete, ChekStatusVosul, ChekStatusReturn, ChekStatusPass, ChekStatusDelay
)
from applications.accounting.serializers import (
    BanksListSerializer, DaftarKolListSerializer, ChekListSerializer,
    HesabDaftarKolListSerializer,
    PardakhtBeTarafHesabListSerializer,
    DaryaftAzTarafHesabListSerializer,
    MinimalPardakht, MinimalDaryaft,
    MinimalBankTransfer,
    MinimalTalabBedeh, MinimalTalabBedehGhest, MinimalPardakht1, MinimalDaryaft1, MinimalChek,
    BankTransferListSerializer, TalabBedehGhestDaily, AllPardakhtBeTarafHesab, AllDaryaftAzTarafHesab, AllTalabBedeh
)
from applications.users.models import User
from applications.utilities.common import (
    RequestParser, EmamParameterError, raise_error, response, get_next_day,
    get_next_month
)
from applications.utilities.permissions import IsICarAdmin
from iCar.constants import NUM_OF_PAGINATION_ITEMS


class DaryaftAzTarafHesabView(APIView):
    permission_classes = [IsAuthenticated, IsICarAdmin]
    authentication_classes = [TokenAuthentication]

    def get(self, request):
        daryaft = DaryaftAzTarafHesab.objects.get(pk=request.query_params['id'])
        return response(
            AllDaryaftAzTarafHesab(daryaft).data
        )

    def post(self, request):
        parser = RequestParser(request.data)
        try:
            pardakht_konande_id = parser.int("pardakht_konande_id", None, is_optional=False)
            tavassot_id = parser.int("tavassot_id", None, is_optional=False)
            tarikh_daryaft = parser.timestamp("tarikh_daryaft", None, is_optional=False)
            jahat = parser.str("jahat", "", is_optional=False)
            babat = parser.str("babat", "", is_optional=True)
            tozihat = parser.str("tozihat", "", is_optional=True)
            hesab_daftar_kol_id = parser.int("hesab_daftar_kol_id", None, is_optional=False)
            pos_amount = parser.int("pos_amount", 0, is_optional=True)
            card_amount = parser.int("card_amount", 0, is_optional=True)
            cash_amount = parser.int("cash_amount", 0, is_optional=True)
            identity = parser.int("id", None, is_optional=True)
            cheks = list(request.data['cheks'])
            ghest_id = request.data.get('ghest_data')
            if ghest_id is not None:
                ghest_id = int(ghest_id)
        except EmamParameterError as e:
            return raise_error("PARAMETERS_ERROR", e.field_name + "|url: accounting/daryaft_az_taraf_hesab/")

        if jahat not in DaryaftAzTarafHesab.JAHATS:
            return raise_error("PARAMETERS_ERROR", "jahat")

        try:
            pardakht_konande = User.objects.get(id=pardakht_konande_id)
        except User.DoesNotExist:
            return raise_error("NOT_FOUND_ERROR", "PARDAKHT_KONANDE")

        try:
            tavassot = User.objects.get(id=tavassot_id)
        except User.DoesNotExist:
            return raise_error("NOT_FOUND_ERROR", "TAVASSOT")

        try:
            hesab_daftar_kol = HesabDaftarKol.objects.get(id=hesab_daftar_kol_id)
        except HesabDaftarKol.DoesNotExist:
            return raise_error("NOT_FOUND_ERROR", "DAFTAR_KOL")

        if identity:
            daryaft_az_taraf_hesab = DaryaftAzTarafHesab.objects.get(pk=identity)
        else:
            daryaft_az_taraf_hesab = DaryaftAzTarafHesab()
        daryaft_az_taraf_hesab.pardakht_konande = pardakht_konande
        daryaft_az_taraf_hesab.tavassot = tavassot
        daryaft_az_taraf_hesab.tarikh_daryaft = tarikh_daryaft
        daryaft_az_taraf_hesab.jahat = jahat
        daryaft_az_taraf_hesab.bAbat = babat
        daryaft_az_taraf_hesab.tozihat = tozihat
        daryaft_az_taraf_hesab.pos_amount = pos_amount or 0
        daryaft_az_taraf_hesab.card_amount = card_amount or 0
        daryaft_az_taraf_hesab.cash = cash_amount or 0
        daryaft_az_taraf_hesab.hesab_daftar_kol = hesab_daftar_kol
        daryaft_az_taraf_hesab.save()
        ch = []
        total_paid_check = 0
        for chek in cheks:
            total_paid_check += chek['amount']
            ch.append(
                Chek(
                    tarikh_sarresid=timezone.datetime.fromtimestamp(int(chek['tarikh'])),
                    bank_id=chek['bank_id'],
                    shomare_chek=chek['number'],
                    dar_vajh=chek['dar_vajh'],
                    code=chek['code'],
                    shobe=chek['shobe'],
                    shomare_hesab=chek['hesab'],
                    saheb_hesab=chek['sub_hesab'],
                    mablagh=chek['amount'],
                    daryafti=daryaft_az_taraf_hesab,
                    is_self=False
                )
            )
            Chek.objects.bulk_create(ch)

        if ghest_id:
            total_paid = total_paid_check
            total_paid += daryaft_az_taraf_hesab.pos_amount
            total_paid += daryaft_az_taraf_hesab.card_amount
            total_paid += daryaft_az_taraf_hesab.cash

            tbg = TalabBedehGhest.objects.get(pk=ghest_id)
            tbg.tarikh_amal = tarikh_daryaft

            if total_paid < tbg.amount:
                amount = tbg.amount - total_paid
                tbg.amount = total_paid
                TalabBedehGhest.objects.filter(
                    talab_bedeh=tbg.talab_bedeh,
                    num__gt=tbg.num
                ).update(num=F('num') + 1)
                TalabBedehGhest.objects.create(
                    talab_bedeh=tbg.talab_bedeh,
                    num=tbg.num + 1,
                    amount=amount,
                    tarikh_daryaft=tbg.tarikh_daryaft,
                    tarikh_amal=None,
                    tozihat=tbg.tozihat
                )
            tbg.tozihat = tozihat
            tbg.save()
            daryaft_az_taraf_hesab.talab_bedeh_ghest = tbg
            daryaft_az_taraf_hesab.tozihat = tozihat + ' (قسط شماره {})'.format(str(tbg.num))
            daryaft_az_taraf_hesab.save()

        return response({"status": "ok"})


class PardakhtBeTarafHesabView(APIView):
    permission_classes = [IsAuthenticated, IsICarAdmin]
    authentication_classes = [TokenAuthentication]

    def get(self, request):
        pardakht = PardakhtBeTarafHesab.objects.get(pk=request.query_params['id'])
        return response(
            AllPardakhtBeTarafHesab(pardakht).data
        )

    def post(self, request):
        parser = RequestParser(request.data)
        try:
            daryaft_konande_id = parser.int("daryaft_konande_id", None, is_optional=False)
            tavassot_id = parser.int("tavassot_id", None, is_optional=False)
            tarikh_pardakht = parser.timestamp("tarikh_daryaft", None, is_optional=False)
            jahat = parser.str("jahat", "", is_optional=False)
            babat = parser.str("babat", "", is_optional=True)
            tozihat = parser.str("tozihat", "", is_optional=True)
            hesab_daftar_kol_id = parser.int("hesab_daftar_kol_id", None, is_optional=False)
            pos_amount = parser.int("pos_amount", 0, is_optional=True)
            carmozd = parser.int("carmozd", 0, is_optional=True)
            card_amount = parser.int("card_amount", 0, is_optional=True)
            cash_amount = parser.int("cash_amount", 0, is_optional=True)
            identity = parser.int("id", None, is_optional=True)
            cheks = list(request.data['cheks'])
            ashkhas_cheks = list(request.data['ashkhas_cheks'])
            ghest_id = request.data.get('ghest_data')
            if ghest_id is not None:
                ghest_id = int(ghest_id)
        except EmamParameterError as e:
            return raise_error("PARAMETERS_ERROR", e.field_name + "|url: accounting/daryaft_az_taraf_hesab/")

        if jahat not in DaryaftAzTarafHesab.JAHATS:
            return raise_error("PARAMETERS_ERROR", "jahat")

        try:
            daryaft_konande = User.objects.get(id=daryaft_konande_id)
        except User.DoesNotExist:
            return raise_error("NOT_FOUND_ERROR", "PARDAKHT_KONANDE")

        try:
            tavassot = User.objects.get(id=tavassot_id)
        except User.DoesNotExist:
            return raise_error("NOT_FOUND_ERROR", "TAVASSOT")

        try:
            hesab_daftar_kol = HesabDaftarKol.objects.get(id=hesab_daftar_kol_id)
        except HesabDaftarKol.DoesNotExist:
            return raise_error("NOT_FOUND_ERROR", "DAFTAR_KOL")

        if identity:
            pardakht_be_taraf_hesab = PardakhtBeTarafHesab.objects.get(pk=identity)
        else:
            pardakht_be_taraf_hesab = PardakhtBeTarafHesab()
        pardakht_be_taraf_hesab.daryaft_konande = daryaft_konande
        pardakht_be_taraf_hesab.tavassot = tavassot
        pardakht_be_taraf_hesab.tarikh_pardakht = tarikh_pardakht
        pardakht_be_taraf_hesab.jahat = jahat
        pardakht_be_taraf_hesab.bAbat = babat
        pardakht_be_taraf_hesab.tozihat = tozihat
        pardakht_be_taraf_hesab.carmozd = carmozd or 0
        pardakht_be_taraf_hesab.pos_amount = pos_amount or 0
        pardakht_be_taraf_hesab.card_amount = card_amount or 0
        pardakht_be_taraf_hesab.cash = cash_amount or 0
        pardakht_be_taraf_hesab.hesab_daftar_kol = hesab_daftar_kol
        pardakht_be_taraf_hesab.save()
        ch = []
        total_paid_check = 0
        for chek in cheks:
            total_paid_check += chek['amount']
            ch.append(
                Chek(
                    tarikh_sarresid=timezone.datetime.fromtimestamp(int(chek['tarikh'])),
                    bank_id=chek['bank_id'],
                    shomare_chek=chek['number'],
                    dar_vajh=chek['dar_vajh'],
                    code=chek['code'],
                    shobe=chek['shobe'],
                    shomare_hesab=chek['hesab'],
                    saheb_hesab=chek['sub_hesab'],
                    mablagh=chek['amount'],
                    pardakhti=pardakht_be_taraf_hesab,
                    is_self=False
                )
            )

        if ch:
            Chek.objects.bulk_create(ch)

        if ashkhas_cheks:
            bk = []
            m = daryaft_konande.first_name + ' ' + daryaft_konande.last_name
            for i in ashkhas_cheks:
                bk.append(
                    ChekStatusDelete(
                        chek_id=i,
                        tozihat='به آقای {} خرج شد'.format(m)
                    )
                )
            ChekStatusDelete.objects.bulk_create(bk)
            total_paid_check += Chek.objects.filter(
                pk__in=ashkhas_cheks
            ).aggregate(s=Sum('mablagh'))['s'] or 0

        if ghest_id:
            total_paid = total_paid_check
            total_paid += pardakht_be_taraf_hesab.pos_amount
            total_paid += pardakht_be_taraf_hesab.card_amount
            total_paid += pardakht_be_taraf_hesab.cash

            tbg = TalabBedehGhest.objects.get(pk=ghest_id)
            tbg.tarikh_amal = tarikh_pardakht

            if total_paid < tbg.amount:
                amount = tbg.amount - total_paid
                tbg.amount = total_paid
                TalabBedehGhest.objects.filter(
                    talab_bedeh=tbg.talab_bedeh,
                    num__gt=tbg.num
                ).update(num=F('num') + 1)
                TalabBedehGhest.objects.create(
                    talab_bedeh=tbg.talab_bedeh,
                    num=tbg.num + 1,
                    amount=amount,
                    tarikh_daryaft=tbg.tarikh_daryaft,
                    tarikh_amal=None,
                    tozihat=tbg.tozihat
                )
            tbg.tozihat = tozihat
            tbg.save()
            pardakht_be_taraf_hesab.talab_bedeh_ghest = tbg
            pardakht_be_taraf_hesab.tozihat = tozihat + ' (قسط شماره {})'.format(str(tbg.num))
            pardakht_be_taraf_hesab.save()

        return response({"status": "ok"})


class BanksView(APIView):
    permission_classes = [IsAuthenticated, IsICarAdmin]
    authentication_classes = [TokenAuthentication]

    def get(self, request):
        return response(
            BanksListSerializer(Bank.objects.all(), many=True).data
        )

    def post(self, request):
        try:
            name = request.data['name']
        except:
            return raise_error("PARAMETERS_ERROR", "|url: accounting/daryaft_az_taraf_hesab/")

        bank = Bank.objects.create(name=name)
        return response(BanksListSerializer(bank).data)


class DafaterKolDetailView(APIView):
    permission_classes = [IsAuthenticated, IsICarAdmin]
    authentication_classes = [TokenAuthentication]
    daftar_id = None

    def _get_pardakht(self, a, b):
        p = PardakhtBeTarafHesab.objects.filter(**a).filter(
            hesab_daftar_kol__daftar_kol_id=self.daftar_id,
            jahat__in=PardakhtBeTarafHesab.DAFTAR_JAHATS
        )
        if b:
            return p.filter(**b)
        return p

    def _get_daryaft(self, a, b):
        p = DaryaftAzTarafHesab.objects.filter(**a).filter(
            hesab_daftar_kol__daftar_kol_id=self.daftar_id,
            jahat__in=DaryaftAzTarafHesab.DAFTAR_JAHATS
        )
        if b:
            return p.filter(**b)
        return p

    def get(self, request, daftar_id):
        self.daftar_id = daftar_id

        az = request.query_params.get('az')
        amel = request.query_params.get('amel')
        tavasot = request.query_params.get('tavasot')
        babat = request.query_params.get('babat')
        tozihat = request.query_params.get('tozihat')
        tarigh = int(request.query_params.get('tarigh', 0))

        filter_list = {}
        filter_list1 = {}
        filter_list2 = {}
        filter_list3 = {
            # 'tarigh': BankTransfer.TARIGH_NAGHDI
        }
        # chekstatuspass__isnull=False
        chek = Chek.objects.filter(
            Q(
                pardakhti__hesab_daftar_kol__daftar_kol_id=self.daftar_id,
                pardakhti__jahat__in=PardakhtBeTarafHesab.DAFTAR_JAHATS
            ) |
            Q(
                daryafti__hesab_daftar_kol__daftar_kol_id=self.daftar_id,
                daryafti__jahat__in=DaryaftAzTarafHesab.DAFTAR_JAHATS
            )
        )
        chek1 = chek.filter(chekstatuspass__isnull=False)
        chek2 = chek.filter(chekstatusvosul__isnull=False)
        yesterday = None
        if az is not None:
            filter_list1['tarikh_pardakht'] = timezone.datetime.fromtimestamp(float(az)).date()
            filter_list2['tarikh_daryaft'] = timezone.datetime.fromtimestamp(float(az)).date()
            filter_list3['amal__date'] = timezone.datetime.fromtimestamp(float(az)).date()
            chek1 = chek1.filter(chekstatuspass__date=timezone.datetime.fromtimestamp(float(az)).date())
            chek2 = chek2.filter(chekstatusvosul__date=timezone.datetime.fromtimestamp(float(az)).date())
            yesterday = (
                    timezone.datetime.fromtimestamp(float(az)) - timezone.timedelta(days=1)
            ).date()
        if amel is not None:
            filter_list1['daryaft_konande_id'] = int(amel)
            filter_list2['pardakht_konande_id'] = int(amel)
            filter_list3['tavasot_id'] = int(amel)
            chek1 = chek1.filter(
                Q(pardakhti__daryaft_konande_id=int(amel)) | Q(daryafti__pardakht_konande_id=int(amel)))
            chek2 = chek2.filter(
                Q(pardakhti__daryaft_konande_id=int(amel)) | Q(daryafti__pardakht_konande_id=int(amel)))
        if tavasot is not None:
            filter_list['tavassot_id'] = int(tavasot)
            filter_list3['tavasot_id'] = int(tavasot)
            chek1 = chek1.filter(Q(pardakhti__tavassot_id=int(tavasot)) | Q(daryafti__tavassot_id=int(tavasot)))
            chek2 = chek2.filter(Q(pardakhti__tavassot_id=int(tavasot)) | Q(daryafti__tavassot_id=int(tavasot)))
        if babat is not None and babat.strip() != '':
            filter_list['bAbat__icontains'] = babat
            filter_list3['babat_icontains'] = babat
            chek1 = chek1.filter(Q(pardakhti__bAbat__icontains=int(babat)) | Q(daryafti__bAbat__icontains=int(babat)))
            chek2 = chek2.filter(Q(pardakhti__bAbat__icontains=int(babat)) | Q(daryafti__bAbat__icontains=int(babat)))
        if tozihat is not None and tozihat.strip() != '':
            filter_list['tozihat__icontains'] = tozihat
            filter_list3['description_icontains'] = tozihat
            chek1 = chek1.filter(
                Q(pardakhti__tozihat__icontains=int(tozihat)) | Q(daryafti__tozihat__icontains=int(tozihat)))
            chek2 = chek2.filter(
                Q(pardakhti__tozihat__icontains=int(tozihat)) | Q(daryafti__tozihat__icontains=int(tozihat)))

        if tarigh == 0:
            pass
        elif tarigh == 1:
            filter_list['pos_amount__gt'] = 0
        elif tarigh == 2:
            filter_list['card_amount__gt'] = 0
        elif tarigh == 4:
            filter_list1['chek__chekstatuspass__isnull'] = False
            filter_list2['chek__chekstatusvosul__isnull'] = False

        t = None
        if yesterday and len(filter_list.keys()) == 0:
            t1 = PardakhtBeTarafHesab.objects.filter(
                hesab_daftar_kol__daftar_kol_id=self.daftar_id,
                tarikh_pardakht__lte=yesterday,
                jahat__in=PardakhtBeTarafHesab.DAFTAR_JAHATS
            ).annotate(sum=F('pos_amount') + F('card_amount')).aggregate(di=Sum('sum'))['di']

            t2 = DaryaftAzTarafHesab.objects.filter(
                hesab_daftar_kol__daftar_kol_id=self.daftar_id,
                tarikh_daryaft__lte=yesterday,
                jahat__in=DaryaftAzTarafHesab.DAFTAR_JAHATS
            ).annotate(sum=F('pos_amount') + F('card_amount')).aggregate(di=Sum('sum'))['di']

            t3 = BankTransfer.objects.filter(
                mabda__daftar_kol_id=self.daftar_id,
                sabt__lte=yesterday
            ).aggregate(di=Sum('mablagh'))['di']

            t4 = BankTransfer.objects.filter(
                maghsad__daftar_kol_id=self.daftar_id,
                sabt__lte=yesterday
            ).aggregate(di=Sum('mablagh'))['di']

            cc = Chek.objects.filter(
                Q(
                    pardakhti__hesab_daftar_kol__daftar_kol_id=self.daftar_id,
                    pardakhti__jahat__in=PardakhtBeTarafHesab.DAFTAR_JAHATS
                ) |
                Q(
                    daryafti__hesab_daftar_kol__daftar_kol_id=self.daftar_id,
                    daryafti__jahat__in=DaryaftAzTarafHesab.DAFTAR_JAHATS
                )
            )

            t5 = cc.filter(
                chekstatusvosul__date__lte=yesterday
            ).aggregate(di=Sum('mablagh'))['di']

            t6 = cc.filter(
                chekstatuspass__date__lte=yesterday
            ).aggregate(di=Sum('mablagh'))['di']

            if t1 is None:
                t1 = 0
            if t2 is None:
                t2 = 0
            if t3 is None:
                t3 = 0
            if t4 is None:
                t4 = 0
            if t5 is None:
                t5 = 0
            if t6 is None:
                t6 = 0

            t = t1 - t2 + t3 - t4 + t5 - t6

        b = MinimalBankTransfer(BankTransfer.objects.filter(
            Q(mabda__daftar_kol_id=self.daftar_id) | Q(maghsad__daftar_kol_id=self.daftar_id)
        ).filter(**filter_list3), many=True, context={'i': self.daftar_id}).data
        p = MinimalPardakht(self._get_pardakht(filter_list, filter_list1), many=True).data
        d = MinimalDaryaft(self._get_daryaft(filter_list, filter_list2), many=True).data
        c = list(
            MinimalChek(chek1, many=True, context={'i': True}).data
        ) + list(
            MinimalChek(chek2, many=True, context={'i': False}).data
        )

        if tarigh == 1:
            b = []
        elif tarigh == 2:
            b = []
        elif tarigh == 3:
            p = d = []
        elif tarigh == 4:
            c = []

        p.extend(d)
        p.extend(b)
        p.extend(c)
        p.sort(key=lambda x: x['created_at'], reverse=True)

        return response({
            'data': p,
            'sum': t
        })

    def delete(self, request, daftar_id):
        try:
            daftar = DaftarKol.objects.get(pk=daftar_id)
        except DaftarKol.DoesNotExist:
            return raise_error("NOT_FOUND_ERROR", "|url: accounting/daryaft_az_taraf_hesab/")

        daftar.delete()
        return response({})


class HesabKolView(APIView):
    permission_classes = [IsAuthenticated, IsICarAdmin]
    authentication_classes = [TokenAuthentication]

    def get(self, request):
        try:
            daftar_kols = list(map(int, str(request.query_params.get('ids')).split(',')))
        except:
            return response([])

        # HESAB DAFTARI
        daftari = {
            'bedehkar': PardakhtBeTarafHesab.objects.filter(
                hesab_daftar_kol__daftar_kol_id__in=daftar_kols
            ).annotate(
                money=F('pos_amount') + F('card_amount') + F('cash') + F('carmozd')
            ).aggregate(
                total_sum=Sum('money'),
            )['total_sum'] or 0,

            'bestankar': DaryaftAzTarafHesab.objects.filter(
                hesab_daftar_kol__daftar_kol_id__in=daftar_kols
            ).annotate(
                money=F('pos_amount') + F('card_amount') + F('cash')
            ).aggregate(
                total_sum=Sum('money'),
            )['total_sum'] or 0
        }

        cheks = {
            'bedehkar': Chek.objects.filter(
                pardakhti__hesab_daftar_kol__daftar_kol_id__in=daftar_kols,
                chekstatusvosul__isnull=True,
                chekstatuspass__isnull=False
            ).aggregate(total_sum=Sum('mablagh'))['total_sum'] or 0,
            'bestankar': Chek.objects.filter(
                daryafti__hesab_daftar_kol__daftar_kol_id__in=daftar_kols,
                chekstatusvosul__isnull=False,
                chekstatuspass__isnull=True
            ).aggregate(total_sum=Sum('mablagh'))['total_sum'] or 0
        }

        sandogh = {
            'bedehkar': DaryaftAzTarafHesab.objects.filter(
                hesab_daftar_kol__daftar_kol_id__in=daftar_kols
            ).aggregate(total_sum=Sum('cash'))['total_sum'] or 0,
            'bestankar': PardakhtBeTarafHesab.objects.filter(
                hesab_daftar_kol__daftar_kol_id__in=daftar_kols
            ).aggregate(total_sum=Sum('cash'))['total_sum'] or 0
        }

        return response({
            'daftari': daftari,
            'checks': cheks,
            'sandogh': sandogh
        })


class DafaterKolView(APIView):
    permission_classes = [IsAuthenticated, IsICarAdmin]
    authentication_classes = [TokenAuthentication]

    def get(self, request):
        return response(
            DaftarKolListSerializer(
                DaftarKol.objects.all(),
                many=True
            ).data
        )

    def post(self, request):
        try:
            name = request.data['name']
        except:
            return raise_error("PARAMETERS_ERROR", "|url: accounting/daryaft_az_taraf_hesab/")

        daftar = DaftarKol.objects.create(name=name)
        return response(DaftarKolListSerializer(daftar).data)

    def put(self, request):
        try:
            identifier = request.data['id']
            name = request.data.get('name')
            add = request.data.get('add')
            remove = request.data.get('remove')
        except:
            return raise_error("PARAMETERS_ERROR", "|url: accounting/daryaft_az_taraf_hesab/")

        try:
            daftar = DaftarKol.objects.get(pk=identifier)
        except DaftarKol.DoesNotExist:
            return raise_error("NOT_FOUND_ERROR", "|url: accounting/daryaft_az_taraf_hesab/")

        if name:
            daftar.name = name
            daftar.save()
        elif add:
            HesabDaftarKol.objects.filter(pk=add).update(daftar_kol=daftar)
        elif remove:
            HesabDaftarKol.objects.filter(pk=remove).update(daftar_kol=None)

        return response({'status': 'ok'})


class HesabDafaterKolView(APIView):
    permission_classes = [IsAuthenticated, IsICarAdmin]
    authentication_classes = [TokenAuthentication]

    def get(self, request):
        banks = Bank.objects.all()
        hesabs = HesabDaftarKol.objects.filter(daftar_kol__isnull=False)
        hesabs_out = []
        for hesab in hesabs:
            hesabs_out.append({
                'id': hesab.id,
                'name': hesab.name,
                'banks': [{
                    'id': hesab.bank.id,
                    'name': hesab.bank.name,
                }],
            })

        bank_out = []
        for bank in banks:
            bank_out.append({
                'id': bank.id,
                'name': bank.name
            })

        hesabs_out.append({
            'id': 0,
            'name': 'بی‌حساب',
            'banks': bank_out,
        })

        return response(hesabs_out)


class HesabDafateriView(APIView):
    permission_classes = [IsAuthenticated, IsICarAdmin]
    authentication_classes = [TokenAuthentication]

    def get(self, request):
        print request.query_params
        amel = int(request.query_params['amel'])
        tavasot = request.query_params.get('tavasot')
        babat = request.query_params.get('babat')
        tozihat = request.query_params.get('tozihat')
        tarigh = int(request.query_params.get('tarigh', 0))

        pardakht = PardakhtBeTarafHesab.objects.filter(
            tavassot_id=amel,
            jahat__in=PardakhtBeTarafHesab.HESAB_JAHATS
        )
        daryaft = DaryaftAzTarafHesab.objects.filter(
            pardakht_konande_id=amel,
            jahat__in=DaryaftAzTarafHesab.HESAB_JAHATS
        )
        chek = Chek.objects.filter(
            Q(
                pardakhti__jahat__in=PardakhtBeTarafHesab.HESAB_JAHATS,
                pardakhti__tavassot_id=amel
            ) |
            Q(
                daryafti__jahat__in=DaryaftAzTarafHesab.HESAB_JAHATS,
                daryafti__pardakht_konande_id=amel
            )
        )
        chek1 = chek.filter(chekstatuspass__isnull=False)
        chek2 = chek.filter(chekstatusvosul__isnull=False)
        tb = TalabBedeh.objects.filter(shakhs_id=amel)
        if tavasot is not None:
            pardakht = pardakht.filter(daryaft_konande_id=tavasot)
            daryaft = daryaft.filter(tavassot_id=tavasot)
            chek1 = chek1.filter(daryafti__tavassot_id=tavasot)
            chek2 = chek2.filter(pardakhti__daryaft_konande_id=tavasot)
        if babat is not None and babat.strip() != '':
            pardakht = pardakht.filter(bAbat__icontains=babat)
            daryaft = daryaft.filter(bAbat__icontains=babat)
            tb = tb.filter(babat__icontains=babat)
            chek1 = chek1.filter(
                Q(daryafti__bAbat__icontains=babat) |
                Q(pardakhti__bAbat__icontains=babat)
            )
            chek2 = chek2.filter(
                Q(daryafti__bAbat__icontains=babat) |
                Q(pardakhti__bAbat__icontains=babat)
            )
        if tozihat is not None and tozihat.strip() != '':
            pardakht = pardakht.filter(tozihat__icontains=tozihat)
            daryaft = daryaft.filter(tozihat__icontains=tozihat)
            tb = tb.filter(tozihat__icontains=tozihat)
            chek1 = chek1.filter(
                Q(daryafti__tozihat__icontains=babat) |
                Q(pardakhti__tozihat__icontains=babat)
            )
            chek2 = chek2.filter(
                Q(daryafti__tozihat__icontains=babat) |
                Q(pardakhti__tozihat__icontains=babat)
            )

        if tarigh == 0:
            pass
        elif tarigh == 1:
            pardakht = pardakht.filter(pos_amount__gt=0)
            daryaft = daryaft.filter(pos_amount__gt=0)
        elif tarigh == 2:
            pardakht = pardakht.filter(card_amount__gt=0)
            daryaft = daryaft.filter(card_amount__gt=0)
        # elif tarigh == 3:
        #     filter_list3['cash__gt'] = 0

        tbg = tb.filter(talabbedehghest__isnull=True)
        tbn = tb.filter(talabbedehghest__isnull=False).distinct()

        p = MinimalPardakht1(pardakht, many=True).data
        d = MinimalDaryaft1(daryaft, many=True).data
        tg = MinimalTalabBedeh(tbg, many=True).data
        ch1 = MinimalChek(chek1, many=True, context={'i': True}).data
        ch2 = MinimalChek(chek2, many=True, context={'i': False}).data
        tn = []

        for i in tbn:
            tn.extend(
                MinimalTalabBedehGhest(i.talabbedehghest_set, many=True).data
            )

        for item in p:
            item.update({"type_of": 0, 'deleted': False})
        for item in d:
            item.update({"type_of": 1, 'deleted': False})
        for item in tg:
            item.update({"type_of": int(item['sign']) + 2})
        for item in tn:
            item.update({"type_of": int(item['sign']) + 2})
        for item in ch1:
            item.update({"type_of": int(item['sign']), 'deleted': False})
        for item in ch2:
            item.update({"type_of": int(item['sign']), 'deleted': False})

        p.extend(d)
        p.extend(tg)
        p.extend(tn)
        p.extend(ch2)
        p.extend(ch1)
        p.sort(key=lambda x: x['created_at'], reverse=False)

        return response(p)


class HesabDafateriGhestView(APIView):
    permission_classes = [IsAuthenticated, IsICarAdmin]
    authentication_classes = [TokenAuthentication]

    def get(self, request):
        az = request.query_params.get('az')
        ta = request.query_params.get('ta')
        amel = request.query_params.get('amel')
        babat = request.query_params.get('babat')
        tozihat = request.query_params.get('tozihat')
        tp = request.query_params.get('tp')
        tb = TalabBedehGhest.objects.filter(
            deleted=False
        ).distinct()

        if amel is not None:
            tb = tb.filter(talab_bedeh__shakhs_id=amel)

        if az is not None:
            tb = tb.filter(
                tarikh_daryaft__gte=timezone.datetime.fromtimestamp(int(az)).date()
            )
        if ta is not None:
            tb = tb.filter(
                tarikh_daryaft__lte=timezone.datetime.fromtimestamp(int(ta)).date()
            )

        if babat is not None and babat.strip() != '':
            tb = tb.filter(talab_bedeh__babat__icontains=babat)

        if tozihat is not None and tozihat.strip() != '':
            tb = tb.filter(talab_bedeh__tozihat__icontains=tozihat)

        if tp is not None:
            tp = int(tp)
            if tp == 0:
                pass
            elif tp == 1:
                tb = tb.filter(
                    tarikh_amal__isnull=True,
                    tarikh_daryaft__lt=timezone.localtime().date()
                )
            elif tp == 2:
                tb = tb.filter(
                    tarikh_amal__isnull=True,
                    tarikh_daryaft=timezone.localtime().date()
                )
            elif tp == 3:
                tb = tb.filter(
                    tarikh_amal__isnull=False
                )
            elif tp == 4:
                tb = tb.filter(
                    tarikh_amal__isnull=True
                )

        tn = TalabBedehGhestDaily(tb.order_by('tarikh_daryaft', 'talab_bedeh', 'id'), many=True).data

        return response(tn)

    def put(self, request):
        idendity = int(request.data['id'])
        all_off = bool(request.data['all'])
        description = request.data['desc']

        tbg = TalabBedehGhest.objects.get(pk=idendity)
        if all_off:
            TalabBedeh.objects.filter(pk=tbg.talab_bedeh_id).update(
                deleted=True
            )
            TalabBedehGhest.objects.filter(
                talab_bedeh_id=tbg.talab_bedeh_id,
                deleted=False
            ).update(
                deleted=True,
                tozihat=description
            )
        else:
            tbg.deleted = True
            tbg.tozihat = description
            tbg.save()

        return response({})


class HesabDafaterKolsView(APIView):
    permission_classes = [IsAuthenticated, IsICarAdmin]
    authentication_classes = [TokenAuthentication]

    def get(self, request):
        q = bool(int(request.query_params['d']))

        if q:
            hesab_daftar_kols = HesabDaftarKol.objects.filter(daftar_kol__isnull=True)
        else:
            hesab_daftar_kols = HesabDaftarKol.objects.filter()

        return response(
            HesabDaftarKolListSerializer(
                hesab_daftar_kols,
                many=True
            ).data
        )

    def post(self, request):
        try:
            name = request.data['name']
            bank = request.data['bank']
            shobe = request.data['shobe']
            code_shobe = request.data['code_shobe']
            shomare_hesab = request.data['shomare_hesab']
            saheb_hesab = request.data['saheb_hesab']
            telephone = request.data['telephone']
            tozihat = request.data['tozihat']
            has_chek = request.data['has_chek']
            has_pos = request.data['has_pos']
            has_card = request.data['has_card']
        except:
            return raise_error("PARAMETERS_ERROR", "|url: accounting/daryaft_az_taraf_hesab/")

        hesab = HesabDaftarKol.objects.create(
            name=name,
            bank_id=bank['id'],
            shobe=shobe,
            code_shobe=code_shobe,
            shomare_hesab=shomare_hesab,
            telephone=telephone,
            tozihat=tozihat,
            has_chek=has_chek,
            has_pos=has_pos,
            saheb_hesab=saheb_hesab,
            has_card=has_card
        )

        return response(
            HesabDaftarKolListSerializer(
                hesab
            ).data
        )

    def put(self, request):
        try:
            identifier = request.data['id']
            name = request.data['name']
            bank = request.data['bank']
            shobe = request.data['shobe']
            code_shobe = request.data['code_shobe']
            shomare_hesab = request.data['shomare_hesab']
            saheb_hesab = request.data['saheb_hesab']
            telephone = request.data['telephone']
            tozihat = request.data['tozihat']
            has_chek = request.data['has_chek']
            has_pos = request.data['has_pos']
            has_card = request.data['has_card']
        except:
            return raise_error("PARAMETERS_ERROR", "|url: accounting/daryaft_az_taraf_hesab/")

        try:
            hesab = HesabDaftarKol.objects.get(pk=identifier)
        except HesabDaftarKol.DoesNotExist:
            return raise_error("NOT_FOUND_ERROR", "|url: accounting/daryaft_az_taraf_hesab/")

        hesab.name = name
        hesab.bank_id = bank['id']
        hesab.shobe = shobe
        hesab.code_shobe = code_shobe
        hesab.shomare_hesab = shomare_hesab
        hesab.saheb_hesab = saheb_hesab
        hesab.telephone = telephone
        hesab.tozihat = tozihat
        hesab.has_chek = has_chek
        hesab.has_pos = has_pos
        hesab.has_card = has_card
        hesab.save()

        return response({'status': 'ok'})


class TalabBedehView(APIView):
    permission_classes = [IsAuthenticated, IsICarAdmin]
    authentication_classes = [TokenAuthentication]
    IS_TALAB = None

    def get(self, request):
        tb = TalabBedeh.objects.get(pk=request.query_params['id'])
        return response(
            AllTalabBedeh(tb).data
        )

    def post(self, request):
        """
        daryAft e list e dafAter e kol
        """
        try:
            d_id = request.data.get('id')
            d_shaks_id = int(request.data['shaks_id'])
            d_babat = request.data['babat']
            tarikh_sabt = int(request.data['tarikh_sabt'])
            d_amount = int(str(request.data['amount']).replace(',', ''))
            d_tozihat = request.data['tozihat']
        except:
            return raise_error("PARAMETERS_ERROR", "|url: accounting/daryaft_az_taraf_hesab/")

        try:
            has_ghest = True
            d_num = int(str(request.data['num']).replace(',', ''))
            d_num_type = bool(int(request.data['num_type']))
            d_qnum = int(str(request.data['qnum']).replace(',', ''))
            d_tarikh_daryaft = int(request.data['tarikh_daryaft'])
        except:
            has_ghest = False
            d_num = None
            d_qnum = None
            d_num_type = None
            d_tarikh_daryaft = None

        if d_tarikh_daryaft:
            dd = timezone.datetime.fromtimestamp(d_tarikh_daryaft)
        else:
            dd = None

        counts = 0
        if d_id:
            tb = TalabBedeh.objects.get(pk=d_id)
            tb.talabbedehghest_set.filter(deleted=False, tarikh_amal=None).delete()
            counts = tb.talabbedehghest_set.count()
        else:
            tb = TalabBedeh()
        tb.shakhs_id = d_shaks_id
        tb.babat = d_babat
        tb.amount = d_amount
        tb.tarikh_sabt = timezone.datetime.fromtimestamp(tarikh_sabt)
        tb.tozihat = d_tozihat
        tb.is_talab = self.IS_TALAB
        tb.num_type = d_num_type
        tb.qnum = d_qnum
        tb.tarikh_daryaft = dd
        tb.save()

        if has_ghest:
            if d_num_type:
                interval = get_next_day
            else:
                interval = get_next_month
            d_num -= counts
            tbg = []
            fdate = timezone.datetime.fromtimestamp(d_tarikh_daryaft).date()
            j = 0
            for i in range(0, d_num):
                while tb.talabbedehghest_set.filter(num=i + 1 + j).exists():
                    j += 1
                tbg.append(
                    TalabBedehGhest(
                        talab_bedeh=tb,
                        num=i + 1 + j,
                        amount=d_amount,
                        tarikh_daryaft=fdate,
                        tozihat=d_tozihat
                    )
                )
                fdate = interval(fdate, d_qnum)
            TalabBedehGhest.objects.bulk_create(tbg)
        return response({'status': 'ok'})


class TalabkarView(TalabBedehView):
    IS_TALAB = True


class BedehkarView(TalabBedehView):
    IS_TALAB = False


class CheksView(APIView):
    permission_classes = [IsAuthenticated, IsICarAdmin]
    authentication_classes = [TokenAuthentication]

    def get(self, request):
        try:
            q_bank = request.query_params.get('bank')
            q_id = request.query_params.get('id')
            q_shomare_chek = request.query_params.get('shomare_chek')
            q_dar_vajh = request.query_params.get('dar_vajh')
            q_code = request.query_params.get('code')
            q_shobe = request.query_params.get('shobe')
            q_shomare_hesab = request.query_params.get('shomare_hesab')
            q_saheb_hesab = request.query_params.get('saheb_hesab')
            q_mablagh = request.query_params.get('mablagh')
            q_for_daryafti = request.query_params.get('for_daryafti')
            q_type_of = request.query_params.get('type_of')
            q_dki = request.query_params.get('dki')
            q_tarikh_sarresid_ta = request.query_params.get('tarikh_sarresid_ta')
            q_tarikh_sarresid_az = request.query_params.get('tarikh_sarresid_az')

            if q_for_daryafti is not None:
                q_for_daryafti = int(q_for_daryafti) == 1
        except:
            return raise_error("PARAMETERS_ERROR", "|url: accounting/daryaft_az_taraf_hesab/")

        if q_id:
            return response(
                ChekListSerializer(Chek.objects.get(pk=q_id), context={'type': False}).data
            )
        filter_list = {}
        if q_bank is not None:
            q_bank = int(q_bank)
            if q_bank != 0:
                filter_list['bank_id'] = q_bank
        if q_shomare_chek is not None:
            filter_list['shomare_chek__icontains'] = q_shomare_chek
        if q_dar_vajh is not None:
            filter_list['dar_vajh__icontains'] = q_dar_vajh
        if q_code is not None:
            filter_list['code__icontains'] = q_code
        if q_shobe is not None:
            filter_list['shobe__icontains'] = q_shobe
        if q_shomare_hesab is not None:
            filter_list['shomare_hesab__icontains'] = q_shomare_hesab
        if q_saheb_hesab is not None:
            filter_list['saheb_hesab__icontains'] = q_saheb_hesab
        if q_mablagh is not None:
            filter_list['mablagh'] = str(q_mablagh).replace(',', '')
        if q_dki is not None:
            if q_for_daryafti:
                filter_list['daryafti__pardakht_konande_id'] = q_dki
            else:
                filter_list['pardakhti__daryaft_konande_id'] = q_dki
        if q_tarikh_sarresid_ta is not None:
            filter_list['tarikh_sarresid__lte'] = timezone.datetime.fromtimestamp(int(q_tarikh_sarresid_ta))
        if q_tarikh_sarresid_az is not None:
            filter_list['tarikh_sarresid__gte'] = timezone.datetime.fromtimestamp(int(q_tarikh_sarresid_az))
        if q_type_of is not None:
            tp = int(q_type_of)
            if tp == 0:
                filter_list['chekstatusvosul__isnull'] = False
            elif tp == 1:
                filter_list['chekstatusreturn__isnull'] = False
            elif tp == 2:
                filter_list['chekstatusdelay__isnull'] = False
            elif tp == 3:
                filter_list['chekstatusdelete__isnull'] = False
            elif tp == 4:
                filter_list['chekstatuspass__isnull'] = False

        q_q = False
        if q_for_daryafti is not None:
            q_q = True
            if q_for_daryafti:
                filter_list['daryafti__isnull'] = False
            else:
                filter_list['pardakhti__isnull'] = False

        if filter_list:
            chek = Chek.objects.filter(**filter_list)
        else:
            chek = Chek.objects.filter(chekstatusdelete__isnull=True).all()
        chek_return = []
        di = 0
        if q_q:
            chek_return = ChekListSerializer(chek.filter(
                (
                        Q(chekstatusreturn__isnull=False) | Q(chekstatusdelay__isnull=False)
                ) & Q(
                    chekstatusvosul__isnull=True,
                    chekstatuspass__isnull=True
                )
            ), many=True, context={'type': True}).data
            di = chek.filter(
                (
                        Q(chekstatusreturn__isnull=False) | Q(chekstatusdelay__isnull=False)
                ) & Q(
                    chekstatusvosul__isnull=True,
                    chekstatuspass__isnull=True
                )
            ).aggregate(di=Sum('mablagh'))['di'] or 0
            chek = chek.exclude(
                (
                        Q(chekstatusreturn__isnull=False) |
                        Q(chekstatusdelay__isnull=False)
                ) & Q(
                    chekstatusvosul__isnull=True,
                    chekstatuspass__isnull=True
                )
            )
            # di = Chek.objects.filter(
            #     chekstatusreturn__isnull=False,
            #     chekstatusdelay__isnull=True,
            #     chekstatusvosul__isnull=True,
            #     chekstatuspass__isnull=True
            # ).aggregate(di=Sum('mablagh'))['di'] or 0

        chekTotal = chek.all()
        paginator = Paginator(
            chek.order_by('-tarikh_sarresid', 'id'),
            NUM_OF_PAGINATION_ITEMS if q_for_daryafti is None else 1000
        )
        page = request.query_params.get("page", 1)

        try:
            cheks = paginator.page(page)
        except PageNotAnInteger:
            cheks = paginator.page(1)
        except EmptyPage:
            cheks = []
        data = ChekListSerializer(cheks, many=True, context={'type': False}).data
        chek_return.sort(key=lambda x: x['tarikh_sarresid'], reverse=False)
        chek_return.extend(data)
        return response({
            "data": chek_return,
            "total": di + (chekTotal.aggregate(di=Sum('mablagh'))['di'] or 0),
            "num_of_pages": paginator.num_pages
        })


class ChekView(APIView):
    permission_classes = [IsAuthenticated, IsICarAdmin]
    authentication_classes = [TokenAuthentication]

    def put(self, request, chek_id):
        try:
            chek = Chek.objects.get(pk=chek_id)
        except Chek.DoesNotExist:
            return raise_error('PARAMETER_ERROR')

        try:
            mablagh = request.data['mablagh'].replace(',', '')
            code = request.data['code']
            dar_vajh = request.data['dar_vajh']
            saheb_hesab = request.data['saheb_hesab']
            shomare_hesab = request.data['shomare_hesab']
            shomare_chek = request.data['shomare_chek']
            tarikh_sarresid = int(request.data['tarikh_sarresid'])
        except (KeyError, ValueError):
            return raise_error('PARAMETER_ERROR')

        chek.mablagh = mablagh
        chek.code = code
        chek.dar_vajh = dar_vajh
        chek.saheb_hesab = saheb_hesab
        chek.shomare_hesab = shomare_hesab
        chek.shomare_chek = shomare_chek
        chek.tarikh_sarresid = timezone.datetime.fromtimestamp(tarikh_sarresid)
        chek.save()

        return response({})

    def post(self, request, chek_id):
        try:
            chek = Chek.objects.get(pk=chek_id)
        except Chek.DoesNotExist:
            return raise_error("PARAMETERS_ERROR")

        action = int(request.data.get('action'))
        data_1 = request.data.get('data_1', '')
        data_2 = request.data.get('data_2', '')
        data_3 = request.data.get('data_3', '')
        data_4 = request.data.get('data_4', '')

        if action == 0:
            # delete
            ChekStatusDelete.objects.create(
                chek=chek,
                tozihat=data_2
            )
        elif action == 1:
            # vosul
            if data_3:
                km = str(data_3).replace(',', '').strip()
            else:
                km = 0
            ChekStatusVosul.objects.create(
                chek=chek,
                tozihat=data_2,
                hesab_id=data_4,
                date=timezone.datetime.fromtimestamp(data_1),
                kar_mozd=km
            )
        elif action == 2:
            # bargasht
            ChekStatusReturn.objects.create(
                chek=chek,
                tozihat=data_2,
                bank=chek.bank,
                date=timezone.datetime.fromtimestamp(data_1),
                kar_mozd=data_3 or 0
            )
        elif action == 3:
            # pass
            ChekStatusPass.objects.create(
                chek=chek,
                tozihat=data_2,
                date=timezone.datetime.fromtimestamp(data_1)
            )
        elif action == 4:
            # delay
            dl = ChekStatusDelay.objects.create(
                chek=chek,
                tozihat=data_2,
                elat=data_1,
                date=timezone.datetime.fromtimestamp(int(data_3))
            )
            o = dl.chek.tarikh_sarresid
            dl.chek.tarikh_sarresid = dl.date
            dl.date = o
            dl.save()
            dl.chek.save()

        return response({})


class SandoghView(APIView):
    permission_classes = [IsAuthenticated, IsICarAdmin]
    authentication_classes = [TokenAuthentication]

    def get(self, request):
        try:
            query_tavasot = int(request.query_params.get('tavasot', 0))
            query_user = int(request.query_params.get('user', 0))
            query_from = int(request.query_params.get('from', 0))
            query_to = int(request.query_params.get('to', 0))
            query_type = int(request.query_params.get('type', 0))
        except (KeyError, ValueError):
            query_tavasot = 0
            query_user = 0
            query_from = 0
            query_to = 0
            query_type = 0

        pardakht = PardakhtBeTarafHesab.objects.filter(
            cash__gt=0
        )
        daryaft = DaryaftAzTarafHesab.objects.filter(
            cash__gt=0
        )
        bt = BankTransfer.objects.all()

        if query_tavasot:
            daryaft = daryaft.filter(tavassot_id=query_tavasot)
            pardakht = pardakht.filter(daryaft_konande_id=query_tavasot)
            bt = bt.filter(tavasot_id=query_tavasot)
        if query_user:
            daryaft = daryaft.filter(pardakht_konande_id=query_user)
            pardakht = pardakht.filter(tavassot_id=query_user)
            bt = bt.filter(tarigh=-1)
        if query_from:
            d = timezone.datetime.fromtimestamp(query_from).date()
            daryaft = daryaft.filter(tarikh_daryaft__gte=d)
            pardakht = pardakht.filter(tarikh_pardakht__gte=d)
            bt = bt.filter(amal__date__gte=d)
        if query_to:
            d = timezone.datetime.fromtimestamp(query_to).date()
            daryaft = daryaft.filter(tarikh_daryaft__lte=d)
            pardakht = pardakht.filter(tarikh_pardakht__lte=d)
            bt = bt.filter(amal__date__lte=d)

        q_daryaft = True
        q_pardakht = True
        if query_type == 1:
            q_daryaft = False
            bt = bt.filter(mabda=None)
        elif query_type == 2:
            q_pardakht = False
            bt = bt.filter(maghsad=None)
        else:
            bt.filter(
                Q(maghsad=None) |
                Q(mabda=None)
            )

        page = request.query_params.get("page", 1)

        pardakht_paginator = []
        p_num_pages = 0
        if q_pardakht:
            pardakht_paginators = Paginator(pardakht.order_by('-tarikh_pardakht', '-id'), 1000)
            p_num_pages = pardakht_paginators.num_pages
            try:
                pardakht_paginator = pardakht_paginators.page(page)
            except:
                pardakht_paginator = []

        daryaft_paginator = []
        d_num_pages = 0
        if q_daryaft:
            daryaft_paginators = Paginator(daryaft.order_by('-tarikh_daryaft', '-id'), 1000)
            d_num_pages = daryaft_paginators.num_pages
            try:
                daryaft_paginator = daryaft_paginators.page(page)
            except:
                daryaft_paginator = []

        bt_paginators = Paginator(bt.order_by('-amal', '-id'), 1000)
        try:
            bt_paginator = bt_paginators.page(page)
            bt_num_pages = bt_paginators.num_pages
        except:
            bt_paginator = []
            bt_num_pages = 0

        total = 0
        output = []
        for i in bt_paginator:
            if i.maghsad:
                total += i.mablagh
            else:
                total -= i.mablagh
            output.append({
                'is_daryaft': None,
                'data': BankTransferListSerializer(i).data
            })
        for i in pardakht_paginator:
            total += i.cash
            output.append({
                'is_daryaft': False,
                'data': PardakhtBeTarafHesabListSerializer(i).data
            })
        for i in daryaft_paginator:
            total -= i.cash
            output.append({
                'is_daryaft': True,
                'data': DaryaftAzTarafHesabListSerializer(i).data
            })

        output.sort(key=lambda x: x['data']['created_at'], reverse=True)
        return response({
            "data": output,
            "total": -total,
            "num_of_pages": max(bt_num_pages, d_num_pages, p_num_pages)
        })


class TransfersView(APIView):
    permission_classes = [IsAuthenticated, IsICarAdmin]
    authentication_classes = [TokenAuthentication]

    def post(self, request):
        parser = RequestParser(request.data)
        try:
            tarigh = parser.int('tarigh', None, is_optional=False)
            mabda_hesab_id = parser.int('mabda_hesab_id', None, is_optional=False)
            maghsad_hesab_id = parser.int('maghsad_hesab_id', None, is_optional=False)
            sabt = parser.int('sabt', None, is_optional=False)
            amal = parser.int('amal', None, is_optional=False)
            mablagh = parser.int('mablagh', None, is_optional=False)
            karmozd = parser.int('karmozd', None, is_optional=False)
            description = parser.str('description', None, is_optional=False)
            babat = parser.str('babat', None, is_optional=False)
            tavasot_id = parser.int('tavasot_id', None, is_optional=False)
        except EmamParameterError as e:
            return raise_error("PARAMETERS_ERROR", e.field_name + "|url: accounting/daryaft_az_taraf_hesab/")

        if mabda_hesab_id:
            mabda = HesabDaftarKol.objects.get(pk=mabda_hesab_id)
        else:
            mabda = None
        if maghsad_hesab_id:
            maghsad = HesabDaftarKol.objects.get(pk=maghsad_hesab_id)
        else:
            maghsad = None

        if not (mabda or maghsad):
            return raise_error("PARAMETERS_ERROR", "|url: accounting/daryaft_az_taraf_hesab/")

        BankTransfer.objects.create(
            tarigh=tarigh,
            sabt=timezone.datetime.fromtimestamp(sabt),
            amal=timezone.datetime.fromtimestamp(amal),
            mablagh=mablagh,
            karmozd=karmozd or 0,
            description=description,
            babat=babat,
            tavasot_id=tavasot_id,
            maghsad=maghsad,
            mabda=mabda
        )

        return response({'status': 'ok'})
