import datetime

import jdatetime
import pytz
from django.core.exceptions import ValidationError
from django.utils.translation import ugettext as _
from django.utils.translation import ugettext_lazy as dutu
from rest_framework.response import Response

from applications.utilities.exceptions import EmamValueError, EmamKeyError

error = {
    # response: {"data": {}, "error": {"status": "not valid parameter", "code": "001"}}
    "PARAMETERS_ERROR": {
        "status": _("not valid parameter"),
        "code": "001"
    },
    # response: {"data": {}, "error": {"status": "object not found", "code": "002"}}
    "NOT_FOUND_ERROR": {
        "status": _("object not found"),
        "code": "002"
    },
    # response: {"data": {}, "error": {"status": "No order found", "code": "003"}}
    "GET_ORDER_ERROR": {
        "status": _("No order found"),
        "code": "003"
    },
    # response: {"data": {}, "error": {"status": "Permission denied", "code": "004"}}
    "PERMISSION_ERROR": {
        "status": _("Permission denied"),
        "code": "004"
    },
    # response: {"data": {}, "error": {"status": "You can do this just one time", "code": "005"}}
    "JUST_ONE_TIME_ERROR": {
        "status": _("You can do this just one time"),
        "code": "005"
    },
    # response: {"data": {}, "error": {"marketer not found", "code": "006"}}
    "DEVICE_ERROR": {
        "status": _("Device Type is not valid"),
        "code": "006"
    },
    # response: {"data": {}, "error": {"status": "you can"t do this for yourself", "code": "007"}}
    "PAYMENT_METHOD_ERROR": {
        "status": _("Payment method is not valid"),
        "code": "007"
    },
    # response: {"data": {}, "error": {"status": "error in psp request", "code": "008"}}
    "PSP_ERROR": {
        "status": _("error in psp request"),
        "code": "008"
    },
    # response: {"data": {}, "error": {"status": "you dont have enough raybon", "code": "009"}}
    "NOT_ENOUGH_RAYBON_ERROR": {
        "status": _("you dont have enough raybon"),
        "code": "009"
    },
    # response: {"data": {}, "error": {"status": "error in buy chip", "code": "010"}}
    "BUY_CHIP_ERROR": {
        "status": _("error in buy chip"),
        "code": "010"
    },
    # response: {"data": {}, "error": {"status": "error in raybon charge", "code": "011"}}
    "RAYBON_CHARGE_ERROR": {
        "status": _("error in raybon charge"),
        "code": "011"
    },
    # response: {"data": {}, "error": {"status": "you dont have enough point", "code": "012"}}
    "NOT_ENOUGH_POINT_ERROR": {
        "status": _("you dont have enough point"),
        "code": "012"
    },
    # response: {"data": {}, "error": {"status": "you can"t do this for yourself", "code": "013"}}
    "For_YOURSELF_ERROR": {
        "status": _("you can't do this for yourself"),
        "code": "013"
    },
    # response: {"data": {}, "error": {"status": "this factor doesn't have customer", "code": "014"}}
    "NO_CUSTOMER_ERROR": {
        "status": _("this factor doesn't have customer"),
        "code": "014"
    },
    # response: {"data": {}, "error": {"status": "this factor paid before", "code": "015"}}
    "PAID_BEFORE_ERROR": {
        "status": _("this factor paid before"),
        "code": "015"
    },
    # response: {"data": {}, "error": {"status": "operation failed in db", "code": "016"}}
    "OPERATION_FAILED_ERROR": {
        "status": _("operation failed in db"),
        "code": "016"
    },
    # response: {"data": {}, "error": {"status": "username or password is not correct", "code": "017"}}
    "AUTH_ERROR": {
        "status": _("username or password is not correct"),
        "code": "017"
    },
    # response: {"data": {}, "error": {"status": "the time of this action has finished or will be started in future", "code": "018"}}
    "TIME_ERROR": {
        "status": _("the time of this action has finished or will be started in future"),
        "code": "018"
    },
    # response: {"data": {}, "error": {"status": "there is no chance for this action at this time", "code": "019"}}
    "NO_CHANCE_ERROR": {
        "status": _("there is no chance for this action at this time"),
        "code": "019"
    },
    # response: {"data": {}, "error": {"status": "there is no participant to do some action on them", "code": "020"}}
    "NO_PARTICIPANT_ERROR": {
        "status": _("there is no participant to do some action on them"),
        "code": "020"
    },
    # response: {"data": {}, "error": {"status": "you can't delete sold products", "code": "021"}}
    "DELETE_SOLD_PRODUCT": {
        "status": _("you can't delete sold products"),
        "code": "021"
    },
    # response: {"data": {}, "error": {"status": "there is no subset to do some action on them", "code": "022"}}
    "NO_SUBSET_ERROR": {
        "status": _("there is no subset to do some action on them"),
        "code": "022"
    },
    # response: {"data": {}, "error": {"status": "hadd e maximum e mojAz barAye in action raAyat nashode ast", "code": "023"}}
    "MAX_ERROR": {
        "status": _("hadd e maximum e mojAz barAye in action raAyat nashode ast"),
        "code": "023"
    },
    # response: {"data": {}, "error": {"status": "you dont have enough chip", "code": "024"}}
    "NOT_ENOUGH_CHIP_ERROR": {
        "status": _("you dont have enough chip"),
        "code": "024"
    },
    # response: {"data": {}, "error": {"status": "there is no store or service for this business", "code": "025"}}
    "NOT_SERVICE_OR_STORE": {
        "status": _("there is no store or service for this business"),
        "code": "025"
    },
    # response: {"data": {}, "error": {"status": "this page is private, you need to follow the user", "code": "026"}}
    "PRIVATE_USER_ERROR": {
        "status": _("this page is private, you need to follow the user"),
        "code": "026"
    },
    # response: {"data": {}, "error": {"status": "this business is luxury and you are not", "code": "027"}}
    "LUXURY_PERMISSION_ERROR": {
        "status": _("this business is luxury and you are not"),
        "code": "027"
    },
    # response: {"data": {}, "error": {"status": "already exist", "code": "028"}}
    "UNIQUE_CONSTRAINT_ERROR": {
        "status": _("already exist"),
        "code": "028"
    },
    # response: {"data": {}, "error": {"status": "business does'nt have eMarket", "code": "029"}}
    "NOT_E-MARKET_ERROR": {
        "status": _("business does'nt have eMarket"),
        "code": "029"
    },
    # response: {"data": {}, "error": {"status": "you selected too much", "code": "030"}}
    "MORE_THAN_AVAILABE": {
        "status": _("you selected too much"),
        "code": "030"
    },
    # response: {"data": {}, "error": {"status": "you selected too much", "code": "030"}}
    "NOT_IN_RANGE": {
        "status": _("selected object is not in allowed range"),
        "code": "031"
    },
    # response: {"data": {}, "error": {"status": "this factor is reserved", "code": "030"}}
    "RESERVED_FACTOR": {
        "status": _("this factor is reserved"),
        "code": "032"
    },
    "FORO0SH_SHOULD_FASKH": {
        "status": _("foro0sh avval bAs faskh she"),
        "code": "033"
    },
}


def raise_error(error_key, item=None):
    if error_key in ["NOT_FOUND_ERROR", "PARAMETERS_ERROR"] and item is not None:
        error_response = error.get(error_key)
        error_response["field"] = item
        return Response(error_response, status=400)

    return Response(error.get(error_key), status=400)


def response(data):
    return Response(data, status=200)


ParameterError = KeyError, ValueError
EmamParameterError = EmamKeyError, EmamValueError


# class RequestParser:
#
#     def __init__(self, request):
#         self.request = request
#
#     def int(self, name, default, is_optional=True):
#
#         if not is_optional:
#             try:
#                 properties = self.request[name]
#             except KeyError as e:
#                 raise RaypoKeyError(message=str(e))
#         else:
#             properties = self.request.get(name, default)
#
#         if properties == default:
#             return properties if default is None else int(properties)
#
#         if properties is None or properties == "":
#             return default
#
#         try:
#             properties = int(properties)
#         except ValueError as e:
#             raise RaypoValueError(message=str(e), field_name=name)
#
#         return properties
#
#     def str(self, name, default=None, is_optional=True):
#         if not is_optional:
#             try:
#                 properties = self.request[name]
#             except KeyError as e:
#                 raise RaypoKeyError(message=str(e))
#         else:
#             properties = self.request.get(name, default)
#
#         if properties is None or properties == "" or properties == default:
#             return default
#
#         return str(properties)
#
#     def float(self, name, default):
#         properties = self.request.get(name, default)
#
#         if properties == default:
#             return properties if default is None else float(properties)
#
#         if properties is None or properties == "":
#             return default
#
#         return float(properties)
#
#     def timestamp(self, key, default=None, is_optional=False):
#         if not is_optional:
#             try:
#                 value = self.request[key]
#             except KeyError as e:
#                 raise RaypoKeyError(message=str(e))
#             except ValueError as e:
#                 raise RaypoValueError(
#                     message=str(e),
#                     field_name=key
#                 )
#         else:
#             value = self.request.get(key, default)
#             if value is None or value == "" or value == default:
#                 return default
#             try:
#                 value = int(value)
#             except ValueError as e:
#                 raise RaypoValueError(
#                     message=str(e),
#                     field_name=key
#                 )
#         return datetime.datetime.fromtimestamp(value)


def validate_percent(value):
    if value < 0 or value > 100:
        raise ValidationError(
            dutu('%(value)s is not valid percent'),
            params={'value': value},
        )


def validate_positive(value):
    if value < 0:
        raise ValidationError(
            dutu('%(value)s is negative'),
            params={'value': value},
        )


class RequestParser:

    def __init__(self, request):
        self.request = request

    def int(self, name, default, is_optional=True):

        if not is_optional:
            try:
                properties = self.request[name]
            except KeyError as e:
                raise EmamKeyError(message=str(e))
        else:
            properties = self.request.get(name, default)

        if properties == default:
            return properties if default is None else int(properties)

        if properties is None or properties == "":
            return default

        try:
            properties = int(str(properties).replace(',', ''))
        except ValueError as e:
            raise EmamValueError(message=str(e), field_name=name)

        return properties

    def str(self, name, default=None, is_optional=True):
        if not is_optional:
            try:
                properties = self.request[name]
            except KeyError as e:
                raise EmamKeyError(message=str(e))
        else:
            properties = self.request.get(name, default)

        if properties is None or properties == "" or properties == default:
            return default

        return unicode(properties)

    def float(self, name, default, is_optional=True):
        if not is_optional:
            try:
                properties = self.request[name]
            except KeyError as e:
                raise EmamKeyError(message=str(e))
        else:
            properties = self.request.get(name, default)

        if properties == default:
            return properties if default is None else float(properties)

        if properties is None or properties == "":
            return default

        return float(properties)

    def timestamp(self, key, default=None, is_optional=True):
        if not is_optional:
            try:
                value = self.request[key]
            except KeyError as e:
                raise EmamKeyError(message=str(e))
            except ValueError as e:
                raise EmamValueError(
                    message=str(e),
                    field_name=key
                )
        else:
            value = self.request.get(key, default)
            if value is None or value == "" or value == default:
                return default
            try:
                value = int(value)
            except ValueError as e:
                raise EmamValueError(
                    message=str(e),
                    field_name=key
                )
        return datetime.datetime.fromtimestamp(int(value), tz=pytz.timezone('Asia/Tehran'))

    def bool(self, name, default=None, is_optional=True):
        if not is_optional:
            try:
                properties = bool(self.request[name])
            except KeyError as e:
                raise EmamKeyError(message=str(e))
        else:
            properties = bool(self.request.get(name, default))

        if properties is None or properties == "" or properties == default:
            return default

        return unicode(properties)


def get_timestamp(date):
    return date.strftime("%s")


def get_month(date):
    jdate = jdatetime.date.fromgregorian(date=date)
    if jdate.month == 12:
        jdate = jdate.replace(month=1, year=jdate.year + 1)
    elif jdate.month == 11 and jdate.day == 30 and not jdate.isleap():
        jdate = jdate.replace(month=12, day=jdate.day - 1)
    elif jdate.month == 6 and jdate.day == 31:
        jdate = jdate.replace(month=7, day=30)
    else:
        jdate = jdate.replace(month=jdate.month + 1)
    return jdate.togregorian()


def get_next_month(date, interval):
    for i in range(interval):
        date = get_month(date)
    return date


def get_next_day(date, interval):
    return date + datetime.timedelta(days=interval)
