# -*- coding: utf-8 -*-
from __future__ import unicode_literals

import datetime

import jdatetime
import pytz
from django.core.paginator import Paginator, EmptyPage
from django.db.models import Q
from rest_framework.authentication import TokenAuthentication
from rest_framework.permissions import IsAuthenticated
from rest_framework.views import APIView

from applications.baygani.models import KharidForo0shCar
from applications.cars.models import Car, CarType, CarSystem
from applications.cars.serializers import CarsListSerializer, ShowCarSerializer
from applications.configs.models import GholnameSettings
from applications.gholname.models import Gholname, KharidNamayeshgahGholname, NaghdiGholname, AghsatiGholname, Faskh, \
    LizingGholname
from applications.gholname.serializers import GholnameSerializer, GhestSerializer, FaskhGholnameSerializer, \
    GholnameListSerializer, FaskhSerializer
from applications.users.models import User
from applications.users.serializers import UserInGholnameSerializer
from applications.utilities.common import response, ParameterError, raise_error, RequestParser, EmamParameterError
from applications.utilities.permissions import IsICarAdmin
from iCar.constants import NUM_OF_PAGINATION_ITEMS


class ShomareSanadView(APIView):
    permission_classes = (IsAuthenticated, IsICarAdmin)
    authentication_classes = (TokenAuthentication,)

    def get(self, request, *args, **kwargs):
        """
        daryAfte shomAre sanad e jadid
        --------

        Response Messages
            - code: 200
            response: {
                "shomare_sanad": 10012
            }

        """
        try:
            shomare_sanad = Gholname.objects.all().order_by("-created_at")[0].shomare_sanad + 1
        except IndexError:
            shomare_sanad = 10000

        return response({"shomare_sanad": shomare_sanad})


class KharidNamayeshgahView(APIView):
    permission_classes = (IsAuthenticated, IsICarAdmin)
    authentication_classes = (TokenAuthentication,)

    def post(self, request, *args, **kwargs):
        """
        ijAd e gholnAme kharid e namAyeshgAh
        --------
        parameters
            - name: name
            type: string

            - name: system
            type: string

            - name: tip
            type: string

            - name: model
            type: string

            - name: color
            type: string

            - name: pelak
            type: string

            - name: shasi
            type: string

            - name: motor
            type: string

            - name: gholname_datetime
            type: int

            - name: tahvil_datetime
            type: int

            - name: total_cost
            type: int

            - name: pish_pardakht
            type: int

            - name: kharj_sanad
            type: int

            - name: dong
            type: string

            - name: saheb_pelak_id
            type: int

            - name: kharidar_id
            type: int

            - name: foro0shande_id
            type: int

            - name: description
            type: string

            - name: shomare_dasti
            type: string

            - name: code
            type: string

            - name: VIN
            type: string

        Response Messages
            - code:200
            response:{
                "car": {
                    "name": "پورشه",
                    "system": "kado0tanbal",
                    "tip": "خوشتیپ",
                    "model": "بالا",
                    "color": "یشمی",
                    "pelak": "123ب21",
                    "shasi": "2345",
                    "motor": "ضعیف"
                },
                "baghimande": 95000,
                "gholname": {
                    "id",
                    "foro0shande": {
                        "first_name": "علی",
                        "last_name": "محمودی",
                        "father": "حسین",
                        "shomare_shenasname": "21234567845",
                        "username": "ali",
                        "sadere": "تهران",
                        "phone_number": "8765434567",
                        "mobile_number": "098765412",
                        "code_posti": "12345678",
                        "address": ""
                    },
                    "kharidar": {
                        "first_name": "علی",
                        "last_name": "محمودی",
                        "father": "حسین",
                        "shomare_shenasname": "21234567845",
                        "username": "ali",
                        "sadere": "تهران",
                        "phone_number": "8765434567",
                        "mobile_number": "098765412",
                        "code_posti": "12345678",
                        "address": ""
                    },
                    "shomare_sanad": 10013,
                    "datetime": "1525693505",
                    "dong": "شش",
                    "total_cost": 100000,
                    "pish_pardakht": 5000,
                    "kharj_sanad": 10000,
                    "tahvil_datetime": "1525693505",
                    "description": "بسیار زیبا"
                },
                "sharayet": "شرایط خوب\r\nشرایط بد"
            }

        Error Messages
            - code: 001
            response: {"status": "not valid parameter", "code": "001"}

            - code: 002
            response: {"status":"object not found","code":"002"}

            - code: 005
            response: {"status":"You can do this just one time", "005"

        """
        try:
            name = request.data["name"]
            system = request.data["system"]
            tip = request.data["tip"]
            model = request.data["model"]
            color = request.data["color"]
            pelak = request.data["pelak"]
            shasi = request.data["shasi"]
            motor = request.data["motor"]
            gholname_datetime = datetime.datetime.fromtimestamp(int(request.data["gholname_datetime"]),
                                                                tz=pytz.timezone('Asia/Tehran'))
            tahvil_datetime = datetime.datetime.fromtimestamp(int(request.data["tahvil_datetime"]),
                                                              tz=pytz.timezone('Asia/Tehran'))
            total_cost = int(request.data["total_cost"])
            pish_pardakht = int(request.data["pish_pardakht"])
            kharj_sanad = int(request.data["kharj_sanad"])
            dong = request.data["dong"]
            saheb_pelak_id = int(request.data["saheb_pelak_id"])
            kharidar_id = int(request.data["kharidar_id"])
            foro0shande_id = int(request.data["foro0shande_id"])
            description = request.data.get("description", "")
            shomare_dasti = int(request.data["shomare_dasti"])
            code = request.data.get("code", "")
            vin = request.data.get("VIN", "")
        except ParameterError:
            return raise_error("PARAMETERS_ERROR")

        try:
            kharidar = User.objects.get(id=kharidar_id)
            foro0shande = User.objects.get(id=foro0shande_id)
            saheb_pelak = User.objects.get(id=saheb_pelak_id)
        except User.DoesNotExist:
            return raise_error("NOT_FOUND_ERROR")

        if Car.objects.filter(
                name=name,
                system=system,
                tip=tip,
                model=model,
                color=color,
                pelak=pelak,
                shasi=shasi,
                motor=motor,
                code=code,
                vin=vin,
                saheb_pelak=saheb_pelak,
                is_in_parking=True
        ).exists():
            return raise_error("JUST_ONE_TIME_ERROR")

        car = Car.objects.create(
            name=name,
            system=system,
            tip=tip,
            model=model,
            color=color,
            pelak=pelak,
            shasi=shasi,
            motor=motor,
            code=code,
            vin=vin,
            saheb_pelak=saheb_pelak,
            is_in_parking=True
        )

        try:
            shomare_sanad = Gholname.objects.all().order_by("-created_at")[0].shomare_sanad + 1
        except IndexError:
            shomare_sanad = 10000

        try:
            gholname_settings = GholnameSettings.objects.all()[0]
        except IndexError:
            gholname_settings = GholnameSettings.objects.create()

        gholname = Gholname.objects.create(
            shomare_sanad=shomare_sanad,
            description=description or '',
            kharidar=kharidar,
            foro0shande=foro0shande,
            datetime=gholname_datetime,
            car=car,
            tahvil_datetime=tahvil_datetime,
            total_cost=total_cost,
            pish_pardakht=pish_pardakht,
            kharj_sanad=kharj_sanad,
            dong=dong,
            shomare_dasti=shomare_dasti,
            sharayet=gholname_settings.sharayet_moamele_naghdi
        )

        KharidForo0shCar.objects.create(gholname_kharid=gholname, pelak_ghadim=gholname.car.pelak)

        KharidNamayeshgahGholname.objects.create(gholname=gholname)
        baghimande = total_cost - pish_pardakht

        return response(
            {
                "gholname": GholnameSerializer(gholname).data,
                "car": CarsListSerializer(car).data,
                "baghimande": baghimande,
                "sharayet": gholname_settings.sharayet_moamele_naghdi
            }
        )


class NaghdiView(APIView):
    permission_classes = (IsAuthenticated, IsICarAdmin)
    authentication_classes = (TokenAuthentication,)

    def post(self, request, *args, **kwargs):
        """
        ijAd e gholnAme naghdi
        --------
        parameters
            - name: car_id
            type: integer

            - name: gholname_datetime
            type: int

            - name: tahvil_datetime
            type: int

            - name: total_cost
            type: int

            - name: pish_pardakht
            type: int

            - name: kharj_sanad
            type: int

            - name: dong
            type: string

            - name: kharidar_id
            type: int

            - name: foro0shande_id
            type: int

            - name: description
            type: string

            - name: main_cost
            type: int

        Response Messages
            - code:200
            response:{
                "car": {
                    "name": "پورشه",
                    "system": "kado0tanbal",
                    "tip": "خوشتیپ",
                    "model": "بالا",
                    "color": "یشمی",
                    "pelak": "123ب21",
                    "shasi": "2345",
                    "motor": "ضعیف"
                },
                "baghimande": 95000,
                "gholname": {
                    "id",
                    "foro0shande": {
                        "first_name": "علی",
                        "last_name": "محمودی",
                        "father": "حسین",
                        "shomare_shenasname": "21234567845",
                        "username": "ali",
                        "sadere": "تهران",
                        "phone_number": "8765434567",
                        "mobile_number": "098765412",
                        "code_posti": "12345678",
                        "address": ""
                    },
                    "kharidar": {
                        "first_name": "علی",
                        "last_name": "محمودی",
                        "father": "حسین",
                        "shomare_shenasname": "21234567845",
                        "username": "ali",
                        "sadere": "تهران",
                        "phone_number": "8765434567",
                        "mobile_number": "098765412",
                        "code_posti": "12345678",
                        "address": ""
                    },
                    "shomare_sanad": 10013,
                    "datetime": "1525693505",
                    "dong": "شش",
                    "total_cost": 100000,
                    "pish_pardakht": 5000,
                    "kharj_sanad": 10000,
                    "tahvil_datetime": "1525693505",
                    "description": "بسیار زیبا"
                },
                "sharayet": "شرایط خوب\r\nشرایط بد"
            }

        Error Messages
            - code: 001
            response: {"status": "not valid parameter", "code": "001"}

            - code: 002
            response: {"status":"object not found","code":"002"}

        """
        try:
            car_id = int(request.data["car_id"])
            gholname_datetime = datetime.datetime.fromtimestamp(int(request.data["gholname_datetime"]),
                                                                tz=pytz.timezone('Asia/Tehran'))
            tahvil_datetime = datetime.datetime.fromtimestamp(int(request.data["tahvil_datetime"]),
                                                              tz=pytz.timezone('Asia/Tehran'))
            total_cost = int(request.data["total_cost"])
            pish_pardakht = int(request.data["pish_pardakht"])
            main_cost = int(request.data["main_cost"])
            kharj_sanad = int(request.data["kharj_sanad"])
            dong = request.data["dong"]
            kharidar_id = int(request.data["kharidar_id"])
            foro0shande_id = int(request.data["foro0shande_id"])
            description = request.data.get("description", "")
        except (KeyError, ValueError):
            return raise_error("PARAMETERS_ERROR")

        try:
            car = Car.objects.get(id=car_id)
        except Car.DoesNotExist:
            return raise_error("NOT_FOUND_ERROR")

        try:
            shomare_sanad = Gholname.objects.all().order_by("-created_at")[0].shomare_sanad + 1
        except IndexError:
            shomare_sanad = 10000

        try:
            kharidar = User.objects.get(id=kharidar_id)
            foro0shande = User.objects.get(id=foro0shande_id)
        except User.DoesNotExist:
            return raise_error("NOT_FOUND_ERROR")

        try:
            gholname_settings = GholnameSettings.objects.all()[0]
        except IndexError:
            gholname_settings = GholnameSettings.objects.create()

        gholname = Gholname.objects.create(
            shomare_sanad=shomare_sanad,
            description=description or '',
            kharidar=kharidar,
            foro0shande=foro0shande,
            datetime=gholname_datetime,
            car=car,
            tahvil_datetime=tahvil_datetime,
            total_cost=total_cost,
            pish_pardakht=pish_pardakht,
            kharj_sanad=kharj_sanad,
            dong=dong,
            sharayet=gholname_settings.sharayet_moamele_naghdi,
            main_cost=main_cost
        )

        NaghdiGholname.objects.create(gholname=gholname)

        try:
            kharid_foro0sh = KharidForo0shCar.objects.get(gholname_kharid__car=gholname.car)
            kharid_foro0sh.gholname_foro0sh = gholname
            kharid_foro0sh.save()
        except KharidForo0shCar.DoesNotExist:
            pass

        baghimande = total_cost - pish_pardakht

        car.is_in_parking = False
        car.save()

        return response(
            {
                "gholname": GholnameSerializer(gholname).data,
                "car": CarsListSerializer(car).data,
                "baghimande": baghimande,
                "sharayet": gholname_settings.sharayet_moamele_naghdi
            }
        )


class AghsatiView(APIView):
    permission_classes = (IsAuthenticated, IsICarAdmin)
    authentication_classes = (TokenAuthentication,)

    def post(self, request, *args, **kwargs):
        """
        ijAd e gholnAme aghsAti
        --------
        parameters
            - name: car_id
            type: integer

            - name: gholname_datetime
            type: int

            - name: tahvil_datetime
            type: int

            - name: total_cost
            type: int

            - name: pish_pardakht
            type: int

            - name: kharj_sanad
            type: int

            - name: main_cost
            type: int

            - name: dong
            type: string

            - name: kharidar_id
            type: int

            - name: foro0shande_id
            type: int

            - name: description
            type: string

            - name: ghest_num
            type: int
            description: tedAd e ghest

            - name: percent_num
            type: int
            description: darsad e so0d

            - name: ghest_start_datetime
            type: int
            description: tArikh e shoro0 e ghest


        Response Messages
            - code:200
            response:{
                "car": {
                    "name": "پورشه",
                    "system": "kado0tanbal",
                    "tip": "خوشتیپ",
                    "model": "بالا",
                    "color": "یشمی",
                    "pelak": "123ب21",
                    "shasi": "2345",
                    "motor": "ضعیف"
                },
                "baghimande": 95000,
                "gholname": {
                    "id",
                    "foro0shande": {
                        "first_name": "علی",
                        "last_name": "محمودی",
                        "father": "حسین",
                        "shomare_shenasname": "21234567845",
                        "username": "ali",
                        "sadere": "تهران",
                        "phone_number": "8765434567",
                        "mobile_number": "098765412",
                        "code_posti": "12345678",
                        "address": ""
                    },
                    "kharidar": {
                        "first_name": "علی",
                        "last_name": "محمودی",
                        "father": "حسین",
                        "shomare_shenasname": "21234567845",
                        "username": "ali",
                        "sadere": "تهران",
                        "phone_number": "8765434567",
                        "mobile_number": "098765412",
                        "code_posti": "12345678",
                        "address": ""
                    },
                    "shomare_sanad": 10013,
                    "datetime": "1525693505",
                    "dong": "شش",
                    "total_cost": 100000,
                    "pish_pardakht": 5000,
                    "kharj_sanad": 10000,
                    "tahvil_datetime": "1525693505",
                    "description": "بسیار زیبا",
                    "num_of_ghest",
                    "percent",
                    "ghest_start_time"
                },
                "mablagh_e_ghest": 10000,
                "sharayet": "شرایط خوب\r\nشرایط بد"
            }

        Error Messages
            - code: 001
            response: {"status": "not valid parameter", "code": "001"}

            - code: 002
            response: {"status":"object not found","code":"002"}

        """
        parser = RequestParser(request.data)
        try:
            car_id = parser.int("car_id", default=None, is_optional=False)
            gholname_datetime = parser.timestamp("gholname_datetime", default=None, is_optional=False)
            tahvil_datetime = parser.timestamp("tahvil_datetime", default=None, is_optional=False)
            total_cost = parser.int("total_cost", default=None, is_optional=False)
            pish_pardakht = parser.int("pish_pardakht", default=None, is_optional=False)
            kharj_sanad = parser.int("kharj_sanad", default=None, is_optional=False)
            dong = parser.str("dong", default=None, is_optional=False)
            kharidar_id = parser.int("kharidar_id", default=None, is_optional=False)
            foro0shande_id = parser.int("foro0shande_id", default=None, is_optional=False)
            description = parser.str("description", default="", is_optional=True)
            ghest_num = parser.int("ghest_num", default=None, is_optional=False)
            percent = parser.float("percent_num", default=None, is_optional=False)
            main_cost = parser.int("main_cost", default=None, is_optional=False)
            ghest_start_datetime = parser.timestamp("ghest_start_datetime", default=None, is_optional=False)
        except EmamParameterError as e:
            return raise_error("PARAMETERS_ERROR", "url:/gholname/aghsati, field:" + e.field_name)

        try:
            car = Car.objects.get(id=car_id)
        except Car.DoesNotExist:
            return raise_error("NOT_FOUND_ERROR")

        try:
            shomare_sanad = Gholname.objects.all().order_by("-created_at")[0].shomare_sanad + 1
        except IndexError:
            shomare_sanad = 10000

        try:
            kharidar = User.objects.get(id=kharidar_id)
            foro0shande = User.objects.get(id=foro0shande_id)
        except User.DoesNotExist:
            return raise_error("NOT_FOUND_ERROR")

        try:
            gholname_settings = GholnameSettings.objects.all()[0]
        except IndexError:
            gholname_settings = GholnameSettings.objects.create()

        gholname = Gholname.objects.create(
            shomare_sanad=shomare_sanad,
            description=description or '',
            kharidar=kharidar,
            foro0shande=foro0shande,
            datetime=gholname_datetime,
            car=car,
            tahvil_datetime=tahvil_datetime,
            total_cost=total_cost,
            pish_pardakht=pish_pardakht,
            kharj_sanad=kharj_sanad,
            dong=dong,
            sharayet=gholname_settings.sharayet_moamele_ghesti,
            main_cost=main_cost
        )
        gholname_aghsati = AghsatiGholname.objects.create(
            gholname=gholname,
            num_of_ghest=ghest_num,
            percent=percent,
            ghest_start_datetime=ghest_start_datetime

        )

        kharid_foro0sh = KharidForo0shCar.objects.get(gholname_kharid__car=gholname.car)
        kharid_foro0sh.gholname_foro0sh = gholname
        kharid_foro0sh.save()

        gholname_aghsati.create_aghsats()

        ghest_cost = gholname_aghsati.calculate_ghest_cost()
        # so0d = gholname_aghsati.calculate_with_so0d()
        baghimande = gholname_aghsati.calculate_with_so0d()
        # baghimande = total_cost - pish_pardakht + so0d

        gholname_serializer = GholnameSerializer(gholname).data
        gholname_serializer["total_cost"] += baghimande + pish_pardakht

        car.is_in_parking = False
        car.save()

        return response(
            {
                "gholname": gholname_serializer,
                "car": CarsListSerializer(car).data,
                "baghimande": baghimande,
                "sharayet": gholname_settings.sharayet_moamele_ghesti,
                "num_of_ghest": ghest_num,
                "percent": percent,
                "ghest_start_time": ghest_start_datetime.strftime("%s"),
                "mablagh_e_ghest": ghest_cost
            }
        )


class GholnameAghsatiView(APIView):
    permission_classes = (IsAuthenticated, IsICarAdmin)
    authentication_classes = (TokenAuthentication,)

    def get(self, request, **kwargs):
        """
        daryAft e gholnAme aghsAti
        --------
        parameters
            - name: gholname_id
            type: integer

        Response Messages
            - code:200
            response:
            agar gholnAme sefr bAshe, yek field be onvAn e "manba_kharid" miad to0 response be in so0rat:

            "manba_kharid" :
            {
                "first_name": "مجید",
                "last_name": "لعل",
                "father": "م",
                "shomare_shenasname": "م",
                "username": "م",
                "sadere": "م",
                "phone_number": "م",
                "mobile_number": "م",
                "code_posti": "م",
                "address": "م"
            }

            {
                "car": {
                    "name": "پورشه",
                    "system": "kado0tanbal",
                    "tip": "خوشتیپ",
                    "model": "بالا",
                    "color": "یشمی",
                    "pelak": "123ب21",
                    "shasi": "2345",
                    "motor": "ضعیف"
                },
                "baghimande": 95000,
                "gholname": {
                    "main_cost": 10000,
                    "total_cost_with_so0d": 120000,
                    "id",
                    "foro0shande": {
                        "first_name": "علی",
                        "last_name": "محمودی",
                        "father": "حسین",
                        "shomare_shenasname": "21234567845",
                        "username": "ali",
                        "sadere": "تهران",
                        "phone_number": "8765434567",
                        "mobile_number": "098765412",
                        "code_posti": "12345678",
                        "address": "",
                        "faskh_info": {"faskh_datetime", "sabt_datetime", "description"}
                    },
                    "kharidar": {
                        "first_name": "علی",
                        "last_name": "محمودی",
                        "father": "حسین",
                        "shomare_shenasname": "21234567845",
                        "username": "ali",
                        "sadere": "تهران",
                        "phone_number": "8765434567",
                        "mobile_number": "098765412",
                        "code_posti": "12345678",
                        "address": ""
                    },
                    "shomare_sanad": 10013,
                    "datetime": "1525693505",
                    "dong": "شش",
                    "total_cost": 100000,
                    "pish_pardakht": 5000,
                    "kharj_sanad": 10000,
                    "tahvil_datetime": "1525693505",
                    "description": "بسیار زیبا",
                    "num_of_ghest",
                    "percent",
                    "ghest_start_time"
                },
                "mablagh_e_ghest": 10000,
                "sharayet": "شرایط خوب\r\nشرایط بد"
            }

        Error Messages
            - code: 001
            response: {"status": "not valid parameter", "code": "001"}

            - code: 002
            response: {"status":"object not found","code":"002"}

        """

        gholname_id = int(kwargs["id"])

        try:
            gholname_aghsati = AghsatiGholname.objects.get(gholname_id=gholname_id)
            gholname = gholname_aghsati.gholname
        except AghsatiGholname.DoesNotExist:
            return raise_error("NOT_FOUND_ERROR")

        ghest_cost = gholname_aghsati.calculate_ghest_cost()
        # so0d = gholname_aghsati.calculate_so0d()
        baghimande = gholname_aghsati.calculate_with_so0d()
        # baghimande = gholname.total_cost - gholname.pish_pardakht + so0d

        try:
            gholname_settings = GholnameSettings.objects.all()[0]
        except IndexError:
            gholname_settings = GholnameSettings.objects.create()

        gholname_serializer = GholnameSerializer(gholname_aghsati.gholname).data
        gholname_serializer["total_cost_with_so0d"] = baghimande + gholname_aghsati.gholname.pish_pardakht

        try:
            gholname_kharid = gholname.foro0sh.gholname_kharid
        except:
            gholname_kharid = None

        kharj_sanad = ""
        if int(gholname.kharj_sanad) == 0:
            kharj_sanad = "خریدار"
        elif int(gholname.kharj_sanad) == 1:
            kharj_sanad = "فروشنده"
        elif int(gholname.kharj_sanad) == 2:
            kharj_sanad = "هر دو"

        jalalian_tahvil_datetime = jdatetime.date.fromgregorian(date=gholname.tahvil_datetime.date())
        date = "{}:{} ، {}/{}/{}".format(gholname.tahvil_datetime.hour, gholname.tahvil_datetime.minute,
                                         jalalian_tahvil_datetime.year, jalalian_tahvil_datetime.month,
                                         jalalian_tahvil_datetime.day)

        serializer = {
            "gholname": gholname_serializer,
            "car": ShowCarSerializer(gholname.car).data,
            "baghimande": baghimande,
            "sharayet": gholname_settings.sharayet_moamele_ghesti.replace("%x", kharj_sanad).replace("%y",
                                                                                                     date).replace("t",
                                                                                                                   str(
                                                                                                                       gholname_aghsati.percent)).replace(
                "z", str(gholname_aghsati.percent - 0.5)),
            "num_of_ghest": gholname_aghsati.num_of_ghest,
            "percent": gholname_aghsati.percent,
            "ghest_start_time": gholname_aghsati.ghest_start_datetime.strftime("%s"),
            "mablagh_e_ghest": ghest_cost
        }

        if gholname.is_sefr and gholname_kharid:
            serializer["manba_kharid"] = UserInGholnameSerializer(gholname_kharid.foro0shande).data

        return response(
            serializer
        )

    def put(self, request, *args, **kwargs):
        """
        virAyesh e gholnAme aghsAti
        --------
        parameters
            -name: id | in url
            type: int
            description: id e gholnAme

            - name: car_id
            type: integer

            - name: gholname_datetime
            type: int

            - name: tahvil_datetime
            type: int

            - name: total_cost
            type: int

            - name: pish_pardakht
            type: int

            - name: kharj_sanad
            type: int

            - name: dong
            type: string

            - name: kharidar_id
            type: int

            - name: foro0shande_id
            type: int

            - name: description
            type: string

            - name: ghest_num
            type: int
            description: tedAd e ghest

            - name: percent_num
            type: int
            description: darsad e so0d

            - name: ghest_start_datetime
            type: int
            description: tArikh e shoro0 e ghest


        Response Messages
            - code:200
            response:{
                "status": "ok"
            }

        Error Messages
            - code: 001
            response: {"status": "not valid parameter", "code": "001"}

            - code: 002
            response: {"status":"object not found","code":"002"}

        """
        parser = RequestParser(request.data)

        gholname_id = int(kwargs["id"])

        try:
            gholname_aghsati = AghsatiGholname.objects.get(gholname_id=gholname_id)
            gholname = gholname_aghsati.gholname
        except AghsatiGholname.DoesNotExist:
            return raise_error("NOT_FOUND_ERROR")

        try:
            car_id = parser.int("car_id", default=None)
            gholname_datetime = parser.timestamp("gholname_datetime", default=None)
            tahvil_datetime = parser.timestamp("tahvil_datetime", default=None)
            total_cost = parser.int("total_cost", default=None)
            pish_pardakht = parser.int("pish_pardakht", default=None)
            kharj_sanad = parser.int("kharj_sanad", default=None)
            dong = parser.str("dong", default=None)
            kharidar_id = parser.int("kharidar_id", default=None)
            foro0shande_id = parser.int("foro0shande_id", default=None)
            description = parser.str("description", default=None)
            ghest_num = parser.int("ghest_num", default=None)
            percent = parser.float("percent_num", default=None)
            ghest_start_datetime = parser.timestamp("ghest_start_datetime", default=None)
        except (KeyError, ValueError):
            return raise_error("PARAMETERS_ERROR", "url:/gholname/aghsati")

        if car_id:
            try:
                car = Car.objects.get(id=car_id)
            except Car.DoesNotExist:
                return raise_error("NOT_FOUND_ERROR")
        else:
            car = None

        if kharidar_id:
            try:
                kharidar = User.objects.get(id=kharidar_id)
            except User.DoesNotExist:
                return raise_error("NOT_FOUND_ERROR")
        else:
            kharidar = None

        if foro0shande_id:
            try:
                foro0shande = User.objects.get(id=foro0shande_id)
            except User.DoesNotExist:
                return raise_error("NOT_FOUND_ERROR")
        else:
            foro0shande = None

        gholname.update(
            description=description or '',
            kharidar=kharidar,
            foro0shande=foro0shande,
            datetime=gholname_datetime,
            car=car,
            tahvil_datetime=tahvil_datetime,
            total_cost=total_cost,
            pish_pardakht=pish_pardakht,
            kharj_sanad=kharj_sanad,
            dong=dong,
        )

        gholname_aghsati.update(
            num_of_ghest=ghest_num,
            percent=percent,
            ghest_start_datetime=ghest_start_datetime

        )

        return response(
            {
                "status": "ok"
            }
        )


class GholnameKharidNamayeshgahView(APIView):
    permission_classes = (IsAuthenticated, IsICarAdmin)
    authentication_classes = (TokenAuthentication,)

    def get(self, request, **kwargs):
        """
        daryAft e gholnAme kharid namayeshgah
        --------
        parameters
            - name: gholname_id
            type: integer

        Response Messages
            - code:200

            mohem: agar gholnAme sefr bAshe response kAmelan shabih be response e gholnAme marbo0t hast masalan
            naghdi age bashe shabihe gholname naghdie, aghsAti age bAshe shabih e gholnAme aghsAtie

            va inke yek field ezAfe mishe be nAm e "manba_kharid" :
            {
                "first_name": "مجید",
                "last_name": "لعل",
                "father": "م",
                "shomare_shenasname": "م",
                "username": "م",
                "sadere": "م",
                "phone_number": "م",
                "mobile_number": "م",
                "code_posti": "م",
                "address": "م"
            }


            response:{
                "car": {
                    "id": 58,
                    "name": "سواری",
                    "system": "پراید",
                    "tip": "هاچبک",
                    "model": "122222",
                    "color": "سبز",
                    "pelak": "11111111",
                    "shasi": "11111",
                    "motor": "11111",
                    "saheb_pelak": {
                        "first_name": "عباس",
                        "last_name": " مخدومی ارضتی",
                        "father": "تدنتد",
                        "code_melli": "ند"
                    },
                    "name_id": 1,
                    "system_id": 1
                },
                "baghimande": 0,
                "gholname": {
                    "main_cost": 10000,
                    "id": 111,
                    "foro0shande": {
                        "first_name": "خلیل",
                        "last_name": "رزاقیان",
                        "father": "حسن",
                        "shomare_shenasname": "1",
                        "username": "1",
                        "sadere": "11",
                        "phone_number": "1",
                        "mobile_number": "1",
                        "code_posti": "1",
                        "address": "1"
                    },
                    "kharidar": {
                        "first_name": "حسن",
                        "last_name": "قاضی خانی",
                        "father": "سیف",
                        "shomare_shenasname": "11",
                        "username": "11",
                        "sadere": "1",
                        "phone_number": "1",
                        "mobile_number": "1",
                        "code_posti": "1",
                        "address": "1"
                    },
                    "shomare_sanad": 10110,
                    "datetime": "1530334589",
                    "dong": "9",
                    "total_cost": 9,
                    "pish_pardakht": 9,
                    "kharj_sanad": 1,
                    "tahvil_datetime": "1530334591",
                    "description": "yyt"
                },
                "sharayet": ""
            }

        Error Messages
            - code: 001
            response: {"status": "not valid parameter", "code": "001"}

            - code: 002
            response: {"status":"object not found","code":"002"}

        """

        gholname_id = int(kwargs["id"])

        try:
            gholname_kharid_namayeshgah = KharidNamayeshgahGholname.objects.get(gholname_id=gholname_id)
            gholname = gholname_kharid_namayeshgah.gholname
        except KharidNamayeshgahGholname.DoesNotExist:
            return raise_error("NOT_FOUND_ERROR")

        baghimande = gholname.total_cost - gholname.pish_pardakht

        try:
            gholname_settings = GholnameSettings.objects.all()[0]
        except IndexError:
            gholname_settings = GholnameSettings.objects.create()

        car = gholname.car

        # TODO: doros she baadan
        try:
            type_id = CarType.objects.get(name=car.name).id
        except CarType.DoesNotExist:
            type_id = None

        # TODO: doros she baadan
        try:
            system_id = CarSystem.objects.get(name=car.system).id
        except CarSystem.DoesNotExist:
            system_id = None
        except CarSystem.MultipleObjectsReturned:
            system_id = CarSystem.objects.filter(name=car.system)[0].id

        car_serializer = CarsListSerializer(gholname.car).data
        car_serializer["name_id"] = type_id
        car_serializer["system_id"] = system_id

        kharj_sanad = ""
        if int(gholname.kharj_sanad) == 0:
            kharj_sanad = "خریدار"
        elif int(gholname.kharj_sanad) == 1:
            kharj_sanad = "فروشنده"
        elif int(gholname.kharj_sanad) == 2:
            kharj_sanad = "هر دو"

        jalalian_tahvil_datetime = jdatetime.date.fromgregorian(date=gholname.tahvil_datetime.date())
        date = "{}:{} ، {}/{}/{}".format(gholname.tahvil_datetime.hour, gholname.tahvil_datetime.minute,
                                         jalalian_tahvil_datetime.year, jalalian_tahvil_datetime.month,
                                         jalalian_tahvil_datetime.day)

        if gholname.is_sefr:
            # gholname.get_type()
            gholname_foro0sh = gholname.kharid.gholname_foro0sh
            gholname_type = gholname_foro0sh.get_type()

            if gholname_type == "sefr_naghdi":
                return response(
                    {
                        "gholname": GholnameSerializer(gholname_foro0sh).data,
                        "manba_kharid": UserInGholnameSerializer(gholname_kharid_namayeshgah.gholname.foro0shande).data,
                        "car": car_serializer,
                        "baghimande": baghimande,
                        "sharayet": gholname_settings.sharayet_moamele_naghdi.replace("%x", kharj_sanad).replace("%y",
                                                                                                                 date),
                    }
                )
            elif gholname_type == "sefr_aghsati":
                gholname_aghsati = gholname_foro0sh.aghsatigholname
                ghest_cost = gholname_aghsati.calculate_ghest_cost()
                # so0d = gholname_aghsati.calculate_so0d()
                baghimande = gholname_aghsati.calculate_with_so0d()
                # baghimande = gholname.total_cost - gholname.pish_pardakht + so0d

                gholname_serializer = GholnameSerializer(gholname_aghsati.gholname).data
                gholname_serializer["total_cost"] += baghimande + gholname_aghsati.gholname.pish_pardakht

                return response(
                    {
                        "gholname": gholname_serializer,
                        "manba_kharid": UserInGholnameSerializer(gholname_kharid_namayeshgah.gholname.foro0shande).data,
                        "car": car_serializer,
                        "baghimande": baghimande,
                        "sharayet": gholname_settings.sharayet_moamele_ghesti.replace("%x", kharj_sanad).replace("%y",
                                                                                                                 date),
                        "num_of_ghest": gholname_aghsati.num_of_ghest,
                        "percent": gholname_aghsati.percent,
                        "ghest_start_time": gholname_aghsati.ghest_start_datetime.strftime("%s"),
                        "mablagh_e_ghest": ghest_cost
                    }
                )
            elif gholname_type == "sefr_lizing":
                gholname_lizing = gholname_foro0sh.lizinggholname
                baghimande = gholname.total_cost - gholname.pish_pardakht

                return response(
                    {
                        "gholname": GholnameSerializer(gholname).data,
                        "manba_kharid": UserInGholnameSerializer(gholname_kharid_namayeshgah.gholname.foro0shande).data,
                        "chek_num": gholname_lizing.chek_num,
                        "moddat_chek": gholname_lizing.moddat_chek,
                        "car": car_serializer,
                        "baghimande": baghimande,
                        "sharayet": gholname_settings.sharayet_moamele_naghdi.replace("%x", kharj_sanad).replace("%y",
                                                                                                                 date),
                    }
                )

        return response(
            {
                "gholname": GholnameSerializer(gholname_kharid_namayeshgah.gholname).data,
                "car": car_serializer,
                "baghimande": baghimande,
                "sharayet": gholname_settings.sharayet_moamele_naghdi.replace("%x", kharj_sanad).replace("%y", date),
            }
        )

    def put(self, request, *args, **kwargs):
        """
        virAyesh e gholnAme kharid e namAyeshgAh
        --------
        parameters
            parameters
            - name: gholname_id
            type: integer

            - name: name
            type: string

            - name: system
            type: string

            - name: tip
            type: string

            - name: model
            type: string

            - name: color
            type: string

            - name: pelak
            type: string

            - name: shasi
            type: string

            - name: motor
            type: string

            - name: gholname_datetime
            type: int

            - name: tahvil_datetime
            type: int

            - name: total_cost
            type: int

            - name: pish_pardakht
            type: int

            - name: kharj_sanad
            type: int

            - name: dong
            type: string

            - name: saheb_pelak_id
            type: int

            - name: kharidar_id
            type: int

            - name: foro0shande_id
            type: int

            - name: description
            type: string

            - name: shomare_dasti
            type: string

            - name: code
            type: string

            - name: VIN
            type: string

        Response Messages
            - code:200
            response:{
                "status": "ok"
            }

        Error Messages
            - code: 001
            response: {"status": "not valid parameter", "code": "001"}

            - code: 002
            response: {"status":"object not found","code":"002"}

            - code: 005
            response: {"status":"You can do this just one time", "005"

        """
        gholname_id = int(kwargs["id"])

        try:
            gholname_kharid_namayeshgah = KharidNamayeshgahGholname.objects.get(gholname_id=gholname_id)
            gholname = gholname_kharid_namayeshgah.gholname
        except KharidNamayeshgahGholname.DoesNotExist:
            return raise_error("NOT_FOUND_ERROR")

        parser = RequestParser(request.data)

        try:
            name = parser.str("name", None)
            system = parser.str("system", None)
            tip = parser.str("tip", None)
            model = parser.str("model", None)
            color = parser.str("color", None)
            pelak = parser.str("pelak", None)
            shasi = parser.str("shasi", None)
            motor = parser.str("motor", None)
            gholname_datetime = parser.timestamp("gholname_datetime", None)
            tahvil_datetime = parser.timestamp("tahvil_datetime", None)
            total_cost = parser.int("total_cost", None)
            pish_pardakht = parser.int("pish_pardakht", None)
            kharj_sanad = parser.int("kharj_sanad", None)
            dong = parser.str("dong", None)
            saheb_pelak_id = parser.int("saheb_pelak_id", None)
            kharidar_id = parser.int("kharidar_id", None)
            foro0shande_id = parser.int("foro0shande_id", None)
            description = parser.str("description", None)
            shomare_dasti = parser.int("shomare_dasti", None)
            code = parser.str("code", None)
            vin = parser.str("VIN", None)
        except ParameterError:
            return raise_error("PARAMETERS_ERROR")

        if kharidar_id:
            try:
                kharidar = User.objects.get(id=kharidar_id)
            except User.DoesNotExist:
                return raise_error("NOT_FOUND_ERROR")
        else:
            kharidar = None

        if foro0shande_id:
            try:
                foro0shande = User.objects.get(id=foro0shande_id)
            except User.DoesNotExist:
                return raise_error("NOT_FOUND_ERROR")
        else:
            foro0shande = None

        if saheb_pelak_id:
            try:
                saheb_pelak = User.objects.get(id=saheb_pelak_id)
            except User.DoesNotExist:
                return raise_error("NOT_FOUND_ERROR")
        else:
            saheb_pelak = None

        gholname.car.update(
            name=name,
            system=system,
            tip=tip,
            model=model,
            color=color,
            pelak=pelak,
            shasi=shasi,
            motor=motor,
            code=code,
            vin=vin,
            saheb_pelak=saheb_pelak
        )

        gholname.update(
            description=description or '',
            kharidar=kharidar,
            foro0shande=foro0shande,
            datetime=gholname_datetime,
            tahvil_datetime=tahvil_datetime,
            total_cost=total_cost,
            pish_pardakht=pish_pardakht,
            kharj_sanad=kharj_sanad,
            dong=dong,
            shomare_dasti=shomare_dasti,
        )

        return response(
            {
                "status": "ok"
            }
        )


class GholnameNaghdiView(APIView):
    permission_classes = (IsAuthenticated, IsICarAdmin)
    authentication_classes = (TokenAuthentication,)

    def get(self, request, **kwargs):
        """
        daryAft e gholnAme naghdi
        --------
        parameters
            - name: gholname_id
            type: integer

        Response Messages
            - code:200
            agar gholnAme sefr bAshe, yek field be onvAn e "manba_kharid" miad to0 response be in so0rat:

            "manba_kharid" :
            {
                "first_name": "مجید",
                "last_name": "لعل",
                "father": "م",
                "shomare_shenasname": "م",
                "username": "م",
                "sadere": "م",
                "phone_number": "م",
                "mobile_number": "م",
                "code_posti": "م",
                "address": "م"
            }

            response:{
                "car": {
                    "name": "پورشه",
                    "system": "kado0tanbal",
                    "tip": "خوشتیپ",
                    "model": "بالا",
                    "color": "یشمی",
                    "pelak": "123ب21",
                    "shasi": "2345",
                    "motor": "ضعیف"
                },
                "baghimande": 95000,
                "gholname": {
                    "main_cost": 10000,
                    "id",
                    "shomare_dasti",
                    "foro0shande": {
                        "first_name": "علی",
                        "last_name": "محمودی",
                        "father": "حسین",
                        "shomare_shenasname": "21234567845",
                        "username": "ali",
                        "sadere": "تهران",
                        "phone_number": "8765434567",
                        "mobile_number": "098765412",
                        "code_posti": "12345678",
                        "address": "",
                        "faskh_info": {"faskh_datetime", "sabt_datetime", "description"}
                    },
                    "kharidar": {
                        "first_name": "علی",
                        "last_name": "محمودی",
                        "father": "حسین",
                        "shomare_shenasname": "21234567845",
                        "username": "ali",
                        "sadere": "تهران",
                        "phone_number": "8765434567",
                        "mobile_number": "098765412",
                        "code_posti": "12345678",
                        "address": ""
                    },
                    "shomare_sanad": 10013,
                    "datetime": "1525693505",
                    "dong": "شش",
                    "total_cost": 100000,
                    "pish_pardakht": 5000,
                    "kharj_sanad": 10000,
                    "tahvil_datetime": "1525693505",
                    "description": "بسیار زیبا"
                },
                "sharayet": "شرایط خوب\r\nشرایط بد"
            }

        Error Messages
            - code: 001
            response: {"status": "not valid parameter", "code": "001"}

            - code: 002
            response: {"status":"object not found","code":"002"}

        """

        gholname_id = int(kwargs["id"])

        try:
            gholname_naghdi = NaghdiGholname.objects.get(gholname_id=gholname_id)
            gholname = gholname_naghdi.gholname
        except NaghdiGholname.DoesNotExist:
            return raise_error("NOT_FOUND_ERROR")

        baghimande = gholname.total_cost - gholname.pish_pardakht

        try:
            gholname_settings = GholnameSettings.objects.all()[0]
        except IndexError:
            gholname_settings = GholnameSettings.objects.create()

        gholname_kharid = None
        if hasattr(gholname, "foro0sh"):
            gholname_kharid = gholname.foro0sh.gholname_kharid

        kharj_sanad = ""
        if int(gholname.kharj_sanad) == 0:
            kharj_sanad = "خریدار"
        elif int(gholname.kharj_sanad) == 1:
            kharj_sanad = "فروشنده"
        elif int(gholname.kharj_sanad) == 2:
            kharj_sanad = "هر دو"

        jalalian_tahvil_datetime = jdatetime.date.fromgregorian(date=gholname.tahvil_datetime.date())
        date = "{}:{} ، {}/{}/{}".format(gholname.tahvil_datetime.hour, gholname.tahvil_datetime.minute,
                                         jalalian_tahvil_datetime.year, jalalian_tahvil_datetime.month,
                                         jalalian_tahvil_datetime.day)

        serializer = {
            "gholname": GholnameSerializer(gholname_naghdi.gholname).data,
            "car": ShowCarSerializer(gholname.car).data,
            "baghimande": baghimande,
            "sharayet": gholname_settings.sharayet_moamele_naghdi.replace("%x", kharj_sanad).replace("%y", date)
        }

        if gholname.is_sefr and gholname_kharid is not None:
            serializer["manba_kharid"] = UserInGholnameSerializer(gholname_kharid.foro0shande).data

        return response(
            serializer
        )

    def put(self, request, **kwargs):
        """
        virAyesh e gholnAme naghdi
        --------
        parameters
            -name: id | in url
            type: int
            description: id e gholnAme

            - name: car_id
            type: integer

            - name: gholname_datetime
            type: int

            - name: tahvil_datetime
            type: int

            - name: total_cost
            type: int

            - name: pish_pardakht
            type: int

            - name: kharj_sanad
            type: int

            - name: dong
            type: string

            - name: kharidar_id
            type: int

            - name: foro0shande_id
            type: int

            - name: description
            type: string

        Response Messages
            - code:200
            response:{"status": "ok"}

        Error Messages
            - code: 001
            response: {"status": "not valid parameter", "code": "001"}

            - code: 002
            response: {"status":"object not found","code":"002"}

        """
        parser = RequestParser(request.data)

        gholname_id = int(kwargs["id"])

        try:
            gholname_naghdi = NaghdiGholname.objects.get(gholname_id=gholname_id)
            gholname = gholname_naghdi.gholname
        except NaghdiGholname.DoesNotExist:
            return raise_error("NOT_FOUND_ERROR")

        try:
            car_id = parser.int("car_id", default=None)
            gholname_datetime = parser.timestamp("gholname_datetime", default=None)
            tahvil_datetime = parser.timestamp("tahvil_datetime", default=None)
            total_cost = parser.int("total_cost", default=None)
            pish_pardakht = parser.int("pish_pardakht", default=None)
            kharj_sanad = parser.int("kharj_sanad", default=None)
            dong = parser.str("dong", default=None)
            kharidar_id = parser.int("kharidar_id", default=None)
            foro0shande_id = parser.int("foro0shande_id", default=None)
            description = parser.str("description", default=None)
        except (KeyError, ValueError):
            return raise_error("PARAMETERS_ERROR")

        if car_id:
            try:
                car = Car.objects.get(id=car_id)
            except Car.DoesNotExist:
                return raise_error("NOT_FOUND_ERROR")
        else:
            car = None

        if kharidar_id:
            try:
                kharidar = User.objects.get(id=kharidar_id)
            except User.DoesNotExist:
                return raise_error("NOT_FOUND_ERROR")
        else:
            kharidar = None

        if foro0shande_id:
            try:
                foro0shande = User.objects.get(id=foro0shande_id)
            except User.DoesNotExist:
                return raise_error("NOT_FOUND_ERROR")
        else:
            foro0shande = None

        gholname.update(
            description or '',
            kharidar=kharidar,
            foro0shande=foro0shande,
            datetime=gholname_datetime,
            car=car,
            tahvil_datetime=tahvil_datetime,
            total_cost=total_cost,
            pish_pardakht=pish_pardakht,
            kharj_sanad=kharj_sanad,
            dong=dong,
        )

        return response(
            {
                "status": "ok"
            }
        )


class GhestsView(APIView):
    permission_classes = (IsAuthenticated, IsICarAdmin)
    authentication_classes = (TokenAuthentication,)

    def get(self, request, **kwargs):
        """
        daryAft e jadval e ghest haye yek gholnAme
        --------
        parameters
            - name: gholname_id
            type: integer

        Response Messages
            - code:200
            response:{
                "aghsat":[
                    {
                        "date": "1525635000",
                        "amount": 13727,
                        "status": "",
                        "tasvie_kamel": 369550
                    },
                    {
                        "date": "1528227000",
                        "amount": 13727,
                        "status": "",
                        "tasvie_kamel": 355823
                    }
                ],
                "num_of_pages": 10,
                "full_name": "abbAs maso0mi",
                "shomare_sanad": 100000,
                "gholname_datetime": 23456787654

            }

        Error Messages
            - code: 001
            response: {"status": "not valid parameter", "code": "001"}

            - code: 002
            response: {"status":"object not found","code":"002"}

        """
        parser = RequestParser(request.query_params)
        try:
            page = parser.int("page", 1)
        except EmamParameterError as e:
            page = 1
        gholname_id = int(kwargs["id"])

        try:
            gholname_aghsati = AghsatiGholname.objects.get(gholname_id=gholname_id)
        except AghsatiGholname.DoesNotExist:
            return raise_error("NOT_FOUND_ERROR")

        aghsat = gholname_aghsati.ghest_set.all().order_by("date")

        paginator = Paginator(aghsat, 18)

        try:
            aghsat = paginator.page(page)
        except EmptyPage:
            aghsat = []

        serializers = GhestSerializer(aghsat, many=True).data

        return response(
            {
                "aghsat": serializers,
                "num_of_pages": paginator.num_pages,
                "full_name": gholname_aghsati.gholname.kharidar.get_full_name(),
                "shomare_sanad": gholname_aghsati.gholname.shomare_sanad,
                "gholname_datetime": gholname_aghsati.gholname.datetime.strftime("%s")
            }
        )


class SearchView(APIView):
    permission_classes = (IsAuthenticated, IsICarAdmin)
    authentication_classes = (TokenAuthentication,)

    def get(self, request, **kwargs):
        """
        search dar gholnAme bar asAs e shomare sanad
        --------
        parameters
            - name: shomare_sanad
            type: integer

        Response Messages
            - code:200
            response:{
                "id": 3,
                "name": "علی رحیمی",
                "father": "قاسم",
                "type": "aghsati", (gozine haye dg: "naghdi", "kharid_namayeshgah")
                "car": {
                    "id": 1,
                    "name": "پورشه",
                    "model": "بالا",
                    "color": "مشکی",
                    "pelak": "21ب321"
                },
                "datetime": "1525680905",
                "tahvil_datetime": "1525680905"
            }

        Error Messages

            - code: 002
            response: {
                "data": {},
                "error": {
                    "status": "object not found",
                    "field": "GHOLNAME",
                    "code": "002"
                }
            }
        """
        shomare_sanad = kwargs["shomare_sanad"]

        try:
            gholname = Gholname.objects.get(shomare_sanad=shomare_sanad)
        except Gholname.DoesNotExist:
            return raise_error("NOT_FOUND_ERROR", item="GHOLNAME")

        return response(FaskhGholnameSerializer(gholname).data)


class FaskhView(APIView):
    permission_classes = (IsAuthenticated, IsICarAdmin)
    authentication_classes = (TokenAuthentication,)

    def get(self, request, **kwargs):
        """
        daryAfte yek gholnAme faskhi
        --------
        parameters
            - name: id
            type: integer
            description: id e gholnAme

        Response Messages
            - code:200
            response:{"faskh_datetime", "sabt_datetime", "description"}

        Error Messages

            - code: 002
            response: {
                "data": {},
                "error": {
                    "status": "object not found",
                    "field": "GHOLNAME",
                    "code": "002"
                }
            }
        """
        gholname_id = kwargs["id"]

        try:
            faskh = Faskh.objects.get(gholname__id=gholname_id)
        except Faskh.DoesNotExist:
            return raise_error("NOT_FOUND_ERROR", item="GHOLNAME")

        serializer = FaskhSerializer(faskh).data

        return response(serializer)

    def post(self, request, **kwargs):
        """
        Faskh e yek gholnAme
        --------
        parameters
            - name: id
            type: integer
            description: id e gholnAme

            - name: faskh_datetime
            type: timestamp
            description: tArikh e faskh

            - name: sabt_datetime
            type: timestamp
            description: tArikh e sabt

            - name: description
            type: string
            description: tozihAt

        Response Messages
            - code:200
            response:{"status": "ok"}

        Error Messages

            - code: 002
            response: {
                "data": {},
                "error": {
                    "status": "object not found",
                    "field": "GHOLNAME",
                    "code": "002"
                }
            }
        """
        gholname_id = kwargs["id"]

        try:
            faskh_datetime = datetime.datetime.fromtimestamp(int(request.data["faskh_datetime"]),
                                                             tz=pytz.timezone('Asia/Tehran'))
            sabt_datetime = datetime.datetime.fromtimestamp(int(request.data["sabt_datetime"]),
                                                            tz=pytz.timezone('Asia/Tehran'))
            description = request.data["description"]
        except ParameterError:
            return raise_error("PARAMETERS_ERROR")

        try:
            gholname = Gholname.objects.get(id=gholname_id, faskh__isnull=True)
        except Gholname.DoesNotExist:
            return raise_error("NOT_FOUND_ERROR", item="GHOLNAME")

        try:
            gholname_foro0sh = gholname.kharid.gholname_foro0sh
            try:
                gholname_foro0sh.faskh
            except AttributeError:
                return raise_error("FORO0SH_SHOULD_FASKH")
        except AttributeError:
            pass

        Faskh.objects.create(
            gholname=gholname,
            faskh_datetime=faskh_datetime,
            sabt_datetime=sabt_datetime,
            description=description or ''
        )

        gholname_type = gholname.get_type()

        if gholname_type == "kharid_namayeshgah":
            gholname.car.is_in_parking = False
        elif gholname_type in ["naghdi", "aghsati"]:
            gholname.car.is_in_parking = True
        else:
            gholname.car.is_in_parking = True

        gholname.car.save()

        return response({"status": "ok"})


class GholnameFilterView(APIView):
    permission_classes = (IsAuthenticated, IsICarAdmin)
    authentication_classes = (TokenAuthentication,)

    def get(self, request):
        parser = RequestParser(request.query_params)
        try:
            page = parser.int("page", 1)
            name = parser.str("name", None)
            username = parser.str("code_melli", None)
            system = parser.str("system", None)
            pelak_ghadim = parser.str("pelak_ghadim", None)
            pelak_jadid = parser.str("pelak_jadid", None)
            motor = parser.str("motor", None)
            shasi = parser.str("shasi", None)
            gholname_type = parser.str("type", None)
        except EmamParameterError as e:
            return raise_error("PARAMETERS_ERROR", e.field_name)

        filter_list = {}

        if system:
            filter_list["car__system__contains"] = system

        if motor:
            filter_list["car__motor__contains"] = motor

        if shasi:
            filter_list["car__shasi__contains"] = shasi

        if pelak_jadid:
            filter_list["car__pelak__contains"] = pelak_jadid

        if gholname_type:
            if gholname_type == "aghsati":
                filter_list["aghsatigholname__isnull"] = False

            elif gholname_type == "naghdi":
                filter_list["naghdigholname__isnull"] = False

            elif gholname_type == "kharid_namayeshgah":
                filter_list["kharidnamayeshgahgholname__isnull"] = False

            elif gholname_type == "is_faskh":
                filter_list["faskh__isnull"] = False

            elif gholname_type == "sefr_naghdi":
                filter_list["naghdigholname__isnull"] = False
                filter_list["is_sefr"] = True

            elif gholname_type == "sefr_aghsati":
                filter_list["aghsatigholname__isnull"] = False
                filter_list["is_sefr"] = True

            elif gholname_type == "sefr_lizing":
                filter_list["lizinggholname__isnull"] = False
                filter_list["is_sefr"] = True

            # TODO: lizing o ejAre ham bAs ezAfe she

        gholnames = Gholname.objects.filter(**filter_list)

        if name:
            gholnames = gholnames.filter(
                Q(kharidar__first_name__contains=name) |
                Q(kharidar__last_name__contains=name) |
                Q(foro0shande__first_name__contains=name) |
                Q(foro0shande__last_name__contains=name)
            )

        if username:
            gholnames = gholnames.filter(
                Q(kharidar__username__contains=username) |
                Q(foro0shande__username__contains=username)
            )

        gholnames = gholnames.order_by("-created_at")

        paginator = Paginator(gholnames, NUM_OF_PAGINATION_ITEMS)

        try:
            gholnames = paginator.page(page)
        except EmptyPage:
            gholnames = []

        serializers = GholnameListSerializer(gholnames, many=True).data

        return response({"gholnames": serializers, "num_of_pages": paginator.num_pages})


class GholnameFilter2View(APIView):
    permission_classes = (IsAuthenticated, IsICarAdmin)
    authentication_classes = (TokenAuthentication,)

    def get(self, request):
        parser = RequestParser(request.query_params)
        try:
            page = parser.int("page", 1)
            name = parser.str("name", None)
            username = parser.str("code_melli", None)
            system = parser.str("system", None)
            pelak_ghadim = parser.str("pelak_ghadim", None)
            pelak_jadid = parser.str("pelak_jadid", None)
            motor = parser.str("motor", None)
            shasi = parser.str("shasi", None)
            type_kharid_namayeshgah = bool(parser.int("type_kharid_namayeshgah", None))
            type_naghdi = bool(parser.int("type_naghdi", None))
            type_aghsati = bool(parser.int("type_aghsati", None))
            type_sefr_aghsati = bool(parser.int("type_sefr_aghsati", None))
            type_sefr_naghdi = bool(parser.int("type_sefr_naghdi", None))
            type_sefr_lizing = bool(parser.int("type_sefr_lizing", None))
            type_is_faskh = bool(parser.int("type_is_faskh", None))
        except EmamParameterError as e:
            return raise_error("PARAMETERS_ERROR", e.field_name)

        filter_list = {}

        if system:
            filter_list["car__system__contains"] = system

        if motor:
            filter_list["car__motor__contains"] = motor

        if shasi:
            filter_list["car__shasi__contains"] = shasi

        filter_list1 = []
        if type_kharid_namayeshgah:
            filter_list1.append(Q(kharidnamayeshgahgholname__isnull=False))
        if type_naghdi:
            filter_list1.append(Q(naghdigholname__isnull=False))
        if type_aghsati:
            filter_list1.append(Q(aghsatigholname__isnull=False))
        if type_sefr_aghsati:
            filter_list1.append(Q(aghsatigholname__isnull=False, is_sefr=True))
        if type_sefr_naghdi:
            filter_list1.append(Q(naghdigholname__isnull=False, is_sefr=True))
        if type_sefr_lizing:
            filter_list1.append(Q(lizinggholname__isnull=False, is_sefr=True))
        if type_is_faskh:
            filter_list1.append(Q(faskh__isnull=False))

        gholnames = Gholname.objects.filter(**filter_list)
        if filter_list1:
            q = filter_list1.pop()
            for i in filter_list1:
                q |= i
            gholnames = gholnames.filter(q)

        if pelak_jadid and pelak_ghadim:
            gholnames = gholnames.filter(
                Q(car__pelak__contains=pelak_ghadim) | Q(car__pelak__contains=pelak_jadid)
            )
        elif pelak_jadid:
            gholnames = gholnames.filter(car__pelak__contains=pelak_jadid)
        elif pelak_ghadim:
            gholnames = gholnames.filter(car__pelak__contains=pelak_ghadim)
        # TODO: lizing o ejAre ham bAs ezAfe she

        if name:
            gholnames = gholnames.filter(
                Q(kharidar__first_name__contains=name) |
                Q(kharidar__last_name__contains=name) |
                Q(foro0shande__first_name__contains=name) |
                Q(foro0shande__last_name__contains=name)
            )

        if username:
            gholnames = gholnames.filter(
                Q(kharidar__username__contains=username) |
                Q(foro0shande__username__contains=username)
            )

        gholnames = gholnames.order_by("-created_at")

        paginator = Paginator(gholnames, NUM_OF_PAGINATION_ITEMS)

        try:
            gholnames = paginator.page(page)
        except EmptyPage:
            gholnames = []

        serializers = GholnameListSerializer(gholnames, many=True).data

        return response({"gholnames": serializers, "num_of_pages": paginator.num_pages})


class SefrNaghdiView(APIView):
    permission_classes = (IsAuthenticated, IsICarAdmin)
    authentication_classes = (TokenAuthentication,)

    def post(self, request, *args, **kwargs):
        """
        ijAd e gholnAme khodro e sefr e naghdi
        --------
        parameters
            - name: name
            type: string

            - name: system
            type: string

            - name: tip
            type: string

            - name: model
            type: string

            - name: color
            type: string

            - name: pelak
            type: string

            - name: shasi
            type: string

            - name: motor
            type: string

            - name: gholname_datetime
            type: int

            - name: tahvil_datetime
            type: int

            - name: total_cost
            type: int

            - name: pish_pardakht
            type: int

            - name: kharj_sanad
            type: int

            - name: main_cost
            type: int

            - name: dong
            type: string

            - name: saheb_pelak_id
            type: int
            optional: True

            - name: kharidar_id
            type: int

            - name: manba_kharid_id
            type: int

            - name: foro0shande_id
            type: int

            - name: description
            type: string

            - name: shomare_dasti
            type: string

            - name: code
            type: string
            optional: True

            - name: VIN
            type: string
            optional: True

        Response Messages
            - code:200
            response:{
                "car": {
                    "name": "پورشه",
                    "system": "kado0tanbal",
                    "tip": "خوشتیپ",
                    "model": "بالا",
                    "color": "یشمی",
                    "pelak": "123ب21",
                    "shasi": "2345",
                    "motor": "ضعیف"
                },
                "baghimande": 95000,
                "gholname": {
                    "id",
                    "foro0shande": {
                        "first_name": "علی",
                        "last_name": "محمودی",
                        "father": "حسین",
                        "shomare_shenasname": "21234567845",
                        "username": "ali",
                        "sadere": "تهران",
                        "phone_number": "8765434567",
                        "mobile_number": "098765412",
                        "code_posti": "12345678",
                        "address": ""
                    },
                    "kharidar": {
                        "first_name": "علی",
                        "last_name": "محمودی",
                        "father": "حسین",
                        "shomare_shenasname": "21234567845",
                        "username": "ali",
                        "sadere": "تهران",
                        "phone_number": "8765434567",
                        "mobile_number": "098765412",
                        "code_posti": "12345678",
                        "address": ""
                    },
                    "shomare_sanad": 10013,
                    "datetime": "1525693505",
                    "dong": "شش",
                    "total_cost": 100000,
                    "pish_pardakht": 5000,
                    "kharj_sanad": 10000,
                    "tahvil_datetime": "1525693505",
                    "description": "بسیار زیبا"
                },
                "sharayet": "شرایط خوب\r\nشرایط بد"
            }

        Error Messages
            - code: 001
            response: {"status": "not valid parameter", "code": "001"}

            - code: 002
            response: {"status":"object not found","code":"002"}

        """
        try:
            name = request.data["name"]
            system = request.data["system"]
            tip = request.data["tip"]
            model = request.data["model"]
            color = request.data["color"]
            pelak = request.data["pelak"]
            shasi = request.data["shasi"]
            motor = request.data["motor"]
            shomare_dasti = int(request.data.get("shomare_dasti", 0))
            gholname_datetime = datetime.datetime.fromtimestamp(int(request.data["gholname_datetime"]),
                                                                tz=pytz.timezone('Asia/Tehran'))
            tahvil_datetime = datetime.datetime.fromtimestamp(int(request.data["tahvil_datetime"]),
                                                              tz=pytz.timezone('Asia/Tehran'))
            total_cost = int(request.data["total_cost"])
            pish_pardakht = int(request.data["pish_pardakht"])
            kharj_sanad = int(request.data["kharj_sanad"])
            dong = request.data["dong"]
            kharidar_id = int(request.data["kharidar_id"])
            foro0shande_id = int(request.data["foro0shande_id"])
            manba_kharid_id = int(request.data["manba_kharid_id"])
            description = request.data.get("description", "")
            main_cost = int(request.data["main_cost"])
        except ParameterError:
            return raise_error("PARAMETERS_ERROR", "url:/gholname/sefr/naghdi")

        is_pelaked = False
        try:
            saheb_pelak_id = int(request.data["saheb_pelak_id"])
            code = request.data["code"]
            vin = request.data["VIN"]
            saheb_pelak = User.objects.get(id=saheb_pelak_id)
            is_pelaked = True
        except (ParameterError, User.DoesNotExist):
            saheb_pelak = None
            code = ""
            vin = ""

        try:
            kharidar = User.objects.get(id=kharidar_id)
            foro0shande = User.objects.get(id=foro0shande_id)
            manba_kharid = User.objects.get(id=manba_kharid_id)
        except User.DoesNotExist:
            return raise_error("NOT_FOUND_ERROR")

        car_properties = {"name": name, "system": system, "tip": tip, "model": model, "color": color, "pelak": pelak,
                          "shasi": shasi, "motor": motor, "is_in_parking": False}

        if is_pelaked:
            car_properties["saheb_pelak"] = saheb_pelak
            car_properties["code"] = code
            car_properties["vin"] = vin

        car = Car.objects.create(**car_properties)

        try:
            shomare_sanad = Gholname.objects.all().order_by("-created_at")[0].shomare_sanad + 1
        except IndexError:
            shomare_sanad = 10000

        try:
            gholname_settings = GholnameSettings.objects.all()[0]
        except IndexError:
            gholname_settings = GholnameSettings.objects.create()

        kharid_namayeshgah_gholname = Gholname.objects.create(
            shomare_sanad=shomare_sanad,
            description=description or '',
            kharidar=foro0shande,
            foro0shande=manba_kharid,
            datetime=gholname_datetime,
            car=car,
            tahvil_datetime=tahvil_datetime,
            total_cost=total_cost,
            pish_pardakht=pish_pardakht,
            kharj_sanad=kharj_sanad,
            dong=dong,
            sharayet=gholname_settings.sharayet_moamele_naghdi,
            is_sefr=True,
            shomare_dasti=shomare_dasti
        )

        KharidNamayeshgahGholname.objects.create(gholname=kharid_namayeshgah_gholname)

        try:
            shomare_sanad = Gholname.objects.all().order_by("-created_at")[0].shomare_sanad + 1
        except IndexError:
            shomare_sanad = 10000

        naghdi_gholname = Gholname.objects.create(
            shomare_sanad=shomare_sanad,
            description=description or '',
            kharidar=kharidar,
            foro0shande=foro0shande,
            datetime=gholname_datetime,
            car=car,
            tahvil_datetime=tahvil_datetime,
            total_cost=total_cost,
            pish_pardakht=pish_pardakht,
            kharj_sanad=kharj_sanad,
            dong=dong,
            sharayet=gholname_settings.sharayet_moamele_naghdi,
            is_sefr=True,
            shomare_dasti=shomare_dasti,
            main_cost=main_cost
        )

        NaghdiGholname.objects.create(gholname=naghdi_gholname)

        KharidForo0shCar.objects.create(
            gholname_kharid=kharid_namayeshgah_gholname,
            gholname_foro0sh=naghdi_gholname
        )

        baghimande = total_cost - pish_pardakht

        return response(
            {
                "gholname": GholnameSerializer(naghdi_gholname).data,
                "car": CarsListSerializer(car).data,
                "baghimande": baghimande,
                "sharayet": gholname_settings.sharayet_moamele_naghdi
            }
        )


class SefrAghsatiView(APIView):
    permission_classes = (IsAuthenticated, IsICarAdmin)
    authentication_classes = (TokenAuthentication,)

    def post(self, request, *args, **kwargs):
        """
        ijAd e gholnAme khodro sefr e aghsati
        --------
        parameters
            - name: name
            type: string

            - name: system
            type: string

            - name: tip
            type: string

            - name: model
            type: string

            - name: color
            type: string

            - name: pelak
            type: string

            - name: main_cost
            type: int

            - name: shasi
            type: string

            - name: motor
            type: string

            - name: gholname_datetime
            type: int

            - name: tahvil_datetime
            type: int

            - name: total_cost
            type: int

            - name: pish_pardakht
            type: int

            - name: kharj_sanad
            type: int

            - name: dong
            type: string

            - name: saheb_pelak_id
            type: int
            optional: True

            - name: kharidar_id
            type: int

            - name: manba_kharid_id
            type: int

            - name: foro0shande_id
            type: int

            - name: description
            type: string

            - name: shomare_dasti
            type: string

            - name: code
            type: string
            optional: True

            - name: VIN
            type: string
            optional: True

            - name: ghest_num
            type: int
            description: tedAd e ghest

            - name: percent_num
            type: int
            description: darsad e so0d

            - name: ghest_start_datetime
            type: int
            description: tArikh e shoro0 e ghest

        Response Messages
            - code:200
            response:{
                "car": {
                    "name": "پورشه",
                    "system": "kado0tanbal",
                    "tip": "خوشتیپ",
                    "model": "بالا",
                    "color": "یشمی",
                    "pelak": "123ب21",
                    "shasi": "2345",
                    "motor": "ضعیف"
                },
                "baghimande": 95000,
                "gholname": {
                    "id",
                    "foro0shande": {
                        "first_name": "علی",
                        "last_name": "محمودی",
                        "father": "حسین",
                        "shomare_shenasname": "21234567845",
                        "username": "ali",
                        "sadere": "تهران",
                        "phone_number": "8765434567",
                        "mobile_number": "098765412",
                        "code_posti": "12345678",
                        "address": ""
                    },
                    "kharidar": {
                        "first_name": "علی",
                        "last_name": "محمودی",
                        "father": "حسین",
                        "shomare_shenasname": "21234567845",
                        "username": "ali",
                        "sadere": "تهران",
                        "phone_number": "8765434567",
                        "mobile_number": "098765412",
                        "code_posti": "12345678",
                        "address": ""
                    },
                    "shomare_sanad": 10013,
                    "datetime": "1525693505",
                    "dong": "شش",
                    "total_cost": 100000,
                    "pish_pardakht": 5000,
                    "kharj_sanad": 10000,
                    "tahvil_datetime": "1525693505",
                    "description": "بسیار زیبا"
                },
                "sharayet": "شرایط خوب\r\nشرایط بد"
            }

        Error Messages
            - code: 001
            response: {"status": "not valid parameter", "code": "001"}

            - code: 002
            response: {"status":"object not found","code":"002"}

        """
        try:
            name = request.data["name"]
            system = request.data["system"]
            tip = request.data["tip"]
            model = request.data["model"]
            color = request.data["color"]
            pelak = request.data["pelak"]
            shasi = request.data["shasi"]
            motor = request.data["motor"]
            gholname_datetime = datetime.datetime.fromtimestamp(int(request.data["gholname_datetime"]),
                                                                tz=pytz.timezone('Asia/Tehran'))
            tahvil_datetime = datetime.datetime.fromtimestamp(int(request.data["tahvil_datetime"]),
                                                              tz=pytz.timezone('Asia/Tehran'))
            total_cost = int(request.data["total_cost"])
            pish_pardakht = int(request.data["pish_pardakht"])
            kharj_sanad = int(request.data["kharj_sanad"])
            dong = request.data["dong"]
            kharidar_id = int(request.data["kharidar_id"])
            foro0shande_id = int(request.data["foro0shande_id"])
            manba_kharid_id = int(request.data["manba_kharid_id"])
            description = request.data.get("description", "")
            ghest_num = int(request.data["ghest_num"])
            percent = float(request.data["percent_num"])
            main_cost = int(request.data["main_cost"])
            ghest_start_datetime = datetime.datetime.fromtimestamp(int(request.data["ghest_start_datetime"]),
                                                                   tz=pytz.timezone('Asia/Tehran'))
        except ParameterError:
            return raise_error("PARAMETERS_ERROR", "url:/gholname/sefr/aghsati/")

        is_pelaked = False
        try:
            saheb_pelak_id = int(request.data["saheb_pelak_id"])
            code = request.data["code"]
            vin = request.data["VIN"]
            saheb_pelak = User.objects.get(id=saheb_pelak_id)
            is_pelaked = True
        except (ParameterError, User.DoesNotExist):
            saheb_pelak = None
            code = ""
            vin = ""

        try:
            kharidar = User.objects.get(id=kharidar_id)
            foro0shande = User.objects.get(id=foro0shande_id)
            manba_kharid = User.objects.get(id=manba_kharid_id)
        except User.DoesNotExist:
            return raise_error("NOT_FOUND_ERROR")

        car_properties = {"name": name, "system": system, "tip": tip, "model": model, "color": color, "pelak": pelak,
                          "shasi": shasi, "motor": motor, "is_in_parking": False}

        if is_pelaked:
            car_properties["saheb_pelak"] = saheb_pelak
            car_properties["code"] = code
            car_properties["vin"] = vin

        car = Car.objects.create(**car_properties)

        try:
            shomare_sanad = Gholname.objects.all().order_by("-created_at")[0].shomare_sanad + 1
        except IndexError:
            shomare_sanad = 10000

        try:
            gholname_settings = GholnameSettings.objects.all()[0]
        except IndexError:
            gholname_settings = GholnameSettings.objects.create()

        kharid_namayeshgah_gholname = Gholname.objects.create(
            shomare_sanad=shomare_sanad,
            description=description or '',
            kharidar=foro0shande,
            foro0shande=manba_kharid,
            datetime=gholname_datetime,
            car=car,
            tahvil_datetime=tahvil_datetime,
            total_cost=total_cost,
            pish_pardakht=pish_pardakht,
            kharj_sanad=kharj_sanad,
            dong=dong,
            sharayet=gholname_settings.sharayet_moamele_naghdi,
            is_sefr=True
        )

        KharidNamayeshgahGholname.objects.create(gholname=kharid_namayeshgah_gholname)

        try:
            shomare_sanad = Gholname.objects.all().order_by("-created_at")[0].shomare_sanad + 1
        except IndexError:
            shomare_sanad = 10000

        gholname = Gholname.objects.create(
            shomare_sanad=shomare_sanad,
            description=description or '',
            kharidar=kharidar,
            foro0shande=foro0shande,
            datetime=gholname_datetime,
            car=car,
            tahvil_datetime=tahvil_datetime,
            total_cost=total_cost,
            pish_pardakht=pish_pardakht,
            kharj_sanad=kharj_sanad,
            dong=dong,
            sharayet=gholname_settings.sharayet_moamele_ghesti,
            is_sefr=True,
            main_cost=main_cost
        )

        gholname_aghsati = AghsatiGholname.objects.create(
            gholname=gholname,
            num_of_ghest=ghest_num,
            percent=percent,
            ghest_start_datetime=ghest_start_datetime

        )

        gholname_aghsati.create_aghsats()

        ghest_cost = gholname_aghsati.calculate_ghest_cost()
        baghimande = gholname_aghsati.calculate_with_so0d()

        # baghimande = total_cost - pish_pardakht + so0d

        KharidForo0shCar.objects.create(
            gholname_kharid=kharid_namayeshgah_gholname,
            gholname_foro0sh=gholname
        )

        gholname_serializer = GholnameSerializer(gholname).data
        gholname_serializer["total_cost"] += baghimande + gholname_aghsati.gholname.pish_pardakht

        return response(
            {
                "gholname": gholname_serializer,
                "car": CarsListSerializer(car).data,
                "baghimande": baghimande,
                "sharayet": gholname_settings.sharayet_moamele_ghesti,
                "num_of_ghest": ghest_num,
                "percent": percent,
                "ghest_start_time": ghest_start_datetime.strftime("%s"),
                "mablagh_e_ghest": ghest_cost
            }
        )


class SefrLizingView(APIView):
    permission_classes = (IsAuthenticated, IsICarAdmin)
    authentication_classes = (TokenAuthentication,)

    def post(self, request, *args, **kwargs):
        """
        ijAd e gholnAme khodro sefr e Lizing
        --------
        parameters
            - name: name
            type: string

            - name: system
            type: string

            - name: tip
            type: string

            - name: model
            type: string

            - name: color
            type: string

            - name: pelak
            type: string
            description: in hatman bAyad 11ک111 bashe

            - name: shasi
            type: string

            - name: main_cost
            type: int

            - name: motor
            type: string

            - name: gholname_datetime
            type: int

            - name: tahvil_datetime
            type: int

            - name: total_cost
            type: int

            - name: pish_pardakht
            type: int

            - name: kharj_sanad
            type: int

            - name: dong
            type: string

            - name: kharidar_id
            type: int

            - name: manba_kharid_id
            type: int

            - name: foro0shande_id
            type: int

            - name: description
            type: string

            - name: chek_num
            type: int
            description: tedAd e chek

            - name: moddat_chek
            type: int
            description: moddat zamAn e har chek, masalan chek e 2 mahe

        Response Messages
            - code:200
            response:{
                "sharayet": "",
                "baghimande": 95000,
                "gholname": {
                    "id": 35,
                    "foro0shande": {
                        "first_name": "علی",
                        "last_name": "رحیمی",
                        "father": "قاسم",
                        "shomare_shenasname": "123456",
                        "username": "ali",
                        "sadere": "لاس وگاس",
                        "phone_number": "12345678",
                        "mobile_number": "09122155717",
                        "code_posti": "345678",
                        "address": "ولنجک"
                    },
                    "kharidar": {
                        "first_name": "علی",
                        "last_name": "رحیمی",
                        "father": "قاسم",
                        "shomare_shenasname": "123456",
                        "username": "ali",
                        "sadere": "لاس وگاس",
                        "phone_number": "12345678",
                        "mobile_number": "09122155717",
                        "code_posti": "345678",
                        "address": "ولنجک"
                    },
                    "shomare_sanad": 10033,
                    "datetime": "1525693505",
                    "dong": "شش",
                    "total_cost": 100000,
                    "pish_pardakht": 5000,
                    "kharj_sanad": 10000,
                    "tahvil_datetime": "1525693505",
                    "description": "بسیار زیبا"
                },
                "car": {
                    "id": 18,
                    "name": "پورشه",
                    "system": "kado0tanbal",
                    "tip": "خوشتیپ",
                    "model": "بالا",
                    "color": "یشمی",
                    "pelak": "123ب21",
                    "shasi": "2345",
                    "motor": "ضعیف"
                },
                "check_num": 10,
                "moddat_chek": 2
            }

        Error Messages
            - code: 001
            response: {"status": "not valid parameter", "code": "001"}

            - code: 002
            response: {"status":"object not found","code":"002"}

        """
        try:
            name = request.data["name"]
            system = request.data["system"]
            tip = request.data["tip"]
            model = request.data["model"]
            color = request.data["color"]
            shasi = request.data["shasi"]
            motor = request.data["motor"]
            gholname_datetime = datetime.datetime.fromtimestamp(int(request.data["gholname_datetime"]),
                                                                tz=pytz.timezone('Asia/Tehran'))
            tahvil_datetime = datetime.datetime.fromtimestamp(int(request.data["tahvil_datetime"]),
                                                              tz=pytz.timezone('Asia/Tehran'))
            total_cost = int(request.data["total_cost"])
            pish_pardakht = int(request.data["pish_pardakht"])
            kharj_sanad = int(request.data["kharj_sanad"])
            dong = request.data["dong"]
            kharidar_id = int(request.data["kharidar_id"])
            foro0shande_id = int(request.data["foro0shande_id"])
            manba_kharid_id = int(request.data["manba_kharid_id"])
            description = request.data.get("description", "")
            chek_num = int(request.data["chek_num"])
            moddat_chek = int(request.data["moddat_chek"])
            main_cost = int(request.data["main_cost"])
        except ParameterError:
            return raise_error("PARAMETERS_ERROR")

        try:
            kharidar = User.objects.get(id=kharidar_id)
            foro0shande = User.objects.get(id=foro0shande_id)
            manba_kharid = User.objects.get(id=manba_kharid_id)
        except User.DoesNotExist:
            return raise_error("NOT_FOUND_ERROR")

        car_properties = {"name": name, "system": system, "tip": tip, "model": model, "color": color,
                          "shasi": shasi, "motor": motor, "is_in_parking": False}

        car = Car.objects.create(**car_properties)

        try:
            shomare_sanad = Gholname.objects.all().order_by("-created_at")[0].shomare_sanad + 1
        except IndexError:
            shomare_sanad = 10000

        try:
            gholname_settings = GholnameSettings.objects.all()[0]
        except IndexError:
            gholname_settings = GholnameSettings.objects.create()

        kharid_namayeshgah_gholname = Gholname.objects.create(
            shomare_sanad=shomare_sanad,
            description=description or '',
            kharidar=foro0shande,
            foro0shande=manba_kharid,
            datetime=gholname_datetime,
            car=car,
            tahvil_datetime=tahvil_datetime,
            total_cost=total_cost,
            pish_pardakht=pish_pardakht,
            kharj_sanad=kharj_sanad,
            dong=dong,
            sharayet=gholname_settings.sharayet_moamele_naghdi,
            is_sefr=True
        )

        KharidNamayeshgahGholname.objects.create(gholname=kharid_namayeshgah_gholname)

        try:
            shomare_sanad = Gholname.objects.all().order_by("-created_at")[0].shomare_sanad + 1
        except IndexError:
            shomare_sanad = 10000

        gholname = Gholname.objects.create(
            shomare_sanad=shomare_sanad,
            description=description or '',
            kharidar=kharidar,
            foro0shande=foro0shande,
            datetime=gholname_datetime,
            car=car,
            tahvil_datetime=tahvil_datetime,
            total_cost=total_cost,
            pish_pardakht=pish_pardakht,
            kharj_sanad=kharj_sanad,
            dong=dong,
            sharayet=gholname_settings.sharayet_moamele_ghesti,
            is_sefr=True,
            main_cost=main_cost
        )

        gholname_lizing = LizingGholname.objects.create(
            gholname=gholname,
            chek_num=chek_num,
            moddat_chek=moddat_chek
        )

        KharidForo0shCar.objects.create(
            gholname_kharid=kharid_namayeshgah_gholname,
            gholname_foro0sh=gholname
        )

        baghimande = total_cost - pish_pardakht

        gholname_serializer = GholnameSerializer(gholname).data

        return response(
            {
                "gholname": gholname_serializer,
                "car": CarsListSerializer(car).data,
                "baghimande": baghimande,
                "sharayet": gholname_settings.sharayet_moamele_naghdi,
                "check_num": chek_num,
                "moddat_chek": moddat_chek,
            }
        )
