# -*- coding: utf-8 -*-
from __future__ import unicode_literals

from django.core.paginator import Paginator, PageNotAnInteger, EmptyPage
from django.db.models import Q
from django.db.models.expressions import Subquery
# Create your views here.
from rest_framework.authentication import TokenAuthentication
from rest_framework.permissions import IsAuthenticated
from rest_framework.views import APIView

from applications.baygani.models import KharidForo0shCar
from applications.cars.models import Car, CarCost, CarType, CarSystem, CarTip, Color
from applications.cars.serializers import CarsListInParkingSerializer, ShowCarSerializer, ShowCarCostSerializer, \
    CarTypesListSerializer, SystemsListSerializer, TipsListSerializer, CarsListInParkingExtendedSerializer
from applications.gholname.models import Gholname
from applications.utilities.common import response, RequestParser, EmamParameterError, raise_error
from applications.utilities.permissions import IsICarAdmin
from iCar.constants import NUM_OF_PAGINATION_ITEMS


class CarsView(APIView):
    serializer_class = CarsListInParkingSerializer
    permission_classes = (IsAuthenticated, IsICarAdmin)
    authentication_classes = (TokenAuthentication,)

    def get(self, request):
        """
        daryAfte list e mashin haye parking ba filtration
        --------
        parameters
            - name: page
            type: int
            description: shomAre pagination -> ?page=1

            - name: name
            type: string
            description: nAm e khodro

            - name: system
            type: string
            description: system e mashin

            - name: shasi
            type: string
            description: shomare shasi e mashin

            - name: model
            type: string
            description: model e mashin


        Response Messages
            - code: 200
            response: {
                "cars": [
                    {
                        "id": 10,
                        "name": "پورشه",
                        "system": "kado0tanbal",
                        "tip": "خوشتیپ",
                        "model": "بالا",
                        "color": "یشمی",
                        "pelak": "123ب21",
                        "shasi": "2345",
                        "motor": "ضعیف",
                        "code": "2345678",
                        "vin": "34567876543456"
                    },
                    {
                        "id": 9,
                        "name": "پورشه",
                        "system": "kado0tanbal",
                        "tip": "خوشتیپ",
                        "model": "بالا",
                        "color": "یشمی",
                        "pelak": "123ب21",
                        "shasi": "2345",
                        "motor": "ضعیف",
                        "code": "2345678",
                        "vin": "34567876543456"
                    },
                    {
                        "id": 1,
                        "name": "پورشه",
                        "system": "قوی",
                        "tip": "خوشتیپ",
                        "model": "بالا",
                        "color": "مشکی",
                        "pelak": "21ب321",
                        "shasi": "345672",
                        "motor": "قوی",
                        "code": "123",
                        "vin": "tgb12y1212"
                    }
                ],
                "num_of_pages": 1
            }
        """
        parser = RequestParser(request.query_params)
        try:
            name = parser.str("name", None)
            system = parser.str("system", None)
            model = parser.str("model", None)
            shasi = parser.str("shasi", None)
        except EmamParameterError as e:
            return raise_error("PARAMETERS_ERROR", e.field_name)

        filter_list = {"is_deleted": False, "is_in_parking": True}

        if system:
            filter_list["system__contains"] = system

        if model:
            filter_list["model__contains"] = model

        if shasi:
            filter_list["shasi__contains"] = shasi

        if name:
            filter_list["name__contains"] = name

        cars = Car.objects.filter(**filter_list).order_by("-created_at")

        paginator = Paginator(cars, NUM_OF_PAGINATION_ITEMS)
        page = request.query_params.get("page")

        try:
            cars = paginator.page(page)
        except PageNotAnInteger:
            cars = paginator.page(1)
        except EmptyPage:
            cars = []

        serializers = self.serializer_class(cars, many=True).data
        return response({"cars": serializers, "num_of_pages": paginator.num_pages})


class CarView(APIView):
    serializer_class = CarsListInParkingSerializer
    permission_classes = (IsAuthenticated, IsICarAdmin)
    authentication_classes = (TokenAuthentication,)

    def get(self, request, **kwargs):
        """
        daryAfte ettelA'At e yek khodro
        --------
        parameters
            - name: car_id
            type: int
            description: id e khodro

        Response Messages
            - code: 200
            response: {
                "id": 10,
                "name": "پورشه",
                "system": "kado0tanbal",
                "tip": "خوشتیپ",
                "name_id": 1,
                "system_id": 2,
                "tip_id": 3,
                "model": "بالا",
                "color": "یشمی",
                "pelak": "123ب21",
                "shasi": "2345",
                "motor": "ضعیف",
                "code": "2345678",
                "vin": "34567876543456",
                "bime_end_time": "1526822833",
                "takhfif_bime": 0,
                "kilometer": 0,
                "moayene_end_time": "1526822833",
                "fuel": 1,
                "color_details": "00000000000",
                "description": "mashin e kho0bie"
            }
        """
        car_id = kwargs["car_id"]

        try:
            car = Car.objects.get(id=car_id)
        except Car.DoesNotExist:
            return raise_error("NOT_FOUND_ERROR", "CAR")

        serializer = ShowCarSerializer(car).data

        return response(serializer)

    def put(self, request, **kwargs):
        """
        virAyeshe ettelA'Ate yek khodro
        --------
        parameters
            - name: car_id
            type: int
            description: id e khodro

            - name: name
            type: string
            description: nAm e khodro

            - name: system
            type: string
            description: system e mashin

            - name: tip
            type: string
            description: tip e mashin

            - name: model
            type: string
            description: model e mashin

            - name: color
            type: string
            description: rang e khodro

            - name: pelak
            type: string
            description: pelak e mashin

            - name: shasi
            type: string
            description: shomare shasi e mashin

            - name: motor
            type: string
            description: motor e mashin

            - name: code
            type: string
            description: bAr code e khodro

            - name: vin
            type: string
            description: vin

            - name: bime_end_time
            type: integer
            description: tArikh e etebAr e bime | timestamp

            - name: takhfif_bime
            type: string
            description: takhfif e bime

            - name: kilometer
            type: string
            description: kArkerd e mashin

            - name: moayene_end_time
            type: string
            description: tArikh e etebAr e moayene fanni | timestamp

            - name: fuel
            type: string
            description: no e so0kht | 1: yegAne so0z, 2: dogAne so0z

            - name: color_details
            type: string
            description: joziAt e rang

            - name: description
            type: string
            description: tozihAt e khodro

        Response Messages
            - code: 200
            response: {"status": "ok"}

        Error Messages
            - code: 002
            response: {"status":"object not found","code":"002"}
        """
        car_id = kwargs["car_id"]
        parser = RequestParser(request.data)

        try:
            name = parser.str("name", None)
            system = parser.str("system", None)
            tip = parser.str("tip", None)
            model = parser.str("model", None)
            color = parser.str("color", None)
            pelak = parser.str("pelak", None)
            shasi = parser.str("shasi", None)
            motor = parser.str("motor", None)
            code = parser.str("code", None)
            vin = parser.str("vin", None)
            bime_end_time = parser.timestamp("bime_end_time", None)
            takhfif_bime = parser.str("takhfif_bime", None)
            kilometer = parser.str("kilometer", None)
            moayene_end_time = parser.timestamp("moayene_end_time", None)
            fuel = parser.str("fuel", None)
            color_details = parser.str("color_details", None)
            description = parser.str("description", None)
        except EmamParameterError as e:
            return raise_error("PARAMETERS_ERROR", e.field_name)

        try:
            car = Car.objects.get(id=car_id)
        except Car.DoesNotExist:
            return raise_error("NOT_FOUND_ERROR", "CAR")

        car.update(name=name, system=system, tip=tip, model=model, color=color, pelak=pelak, shasi=shasi, motor=motor,
                   code=code, vin=vin, bime_end_time=bime_end_time, takhfif_bime=takhfif_bime, kilometer=kilometer,
                   moayene_end_time=moayene_end_time, fuel=fuel, color_details=color_details, description=description)

        return response({"status": "ok"})

    def post(self, request, **kwargs):
        """
        afzo0dan e yek hazine be khodro
        --------
        parameters
            - name: car_id
            type: int
            description: id e khodro

            - name: subject
            type: string
            description: onvAn e hazine

            - name: amount
            type: integer
            description: mablagh e hazine

            - name: store
            type: string
            description: nAm e foro0shgAh ya mechanic e morede nazar

            - name: description
            type: string
            description: tozih

        Response Messages
            - code: 200
            response: {"status": "ok"}

        Error Messages
            - code: 002
            response: {"status":"object not found","code":"002"}
        """
        car_id = kwargs["car_id"]
        parser = RequestParser(request.data)

        try:
            subject = parser.str("subject", default=None, is_optional=False)
            amount = parser.int("amount", default=None, is_optional=False)
            store = parser.str("store", default=None, is_optional=False)
            description = parser.str("description", default=None, is_optional=False)
        except EmamParameterError as e:
            return raise_error("PARAMETERS_ERROR", e.field_name)

        try:
            car = Car.objects.get(id=car_id)
        except Car.DoesNotExist:
            return raise_error("NOT_FOUND_ERROR", "CAR")

        CarCost.objects.create(
            car=car,
            subject=subject,
            amount=amount,
            store=store,
            description=description
        )

        return response({"status": "ok"})


class CarsHistoryView(APIView):
    serializer_class = CarsListInParkingExtendedSerializer
    permission_classes = (IsAuthenticated, IsICarAdmin)
    authentication_classes = (TokenAuthentication,)

    def get(self, request):
        """
        daryAfte tArikhche mashin haye parking ba filtration
        --------
        parameters
            - name: page
            type: int
            description: shomAre pagination -> ?page=1

            - name: name
            type: string
            description: nAm e kharidAr ya foro0shande ha -> ?name=abbas

            - name: system
            type: string
            description: system e mashin

            - name: model
            type: string
            description: model e mashin

            - name: code_melli
            type: string
            description: code melli e kharidAr ya foro0shande -> ?code_melli=1123

            - name: pelak
            type: string
            description: shomAre pelAk e mashin -> ?pelak=

            - name: shasi
            type: string
            description: shomare shasi e mashin -> ?shasi=

            - name: datetime
            type: timestamp
            description: tArikh e morede nazar -> ?datetime=


        Response Messages
            - code: 200
            response: {
                "cars": [
                    {
                        "id": 10,
                        "name": "پورشه",
                        "system": "kado0tanbal",
                        "tip": "خوشتیپ",
                        "model": "بالا",
                        "color": "یشمی",
                        "pelak": "123ب21",
                        "shasi": "2345",
                        "motor": "ضعیف",
                        "code": "2345678",
                        "vin": "34567876543456"
                    },
                    {
                        "id": 9,
                        "name": "پورشه",
                        "system": "kado0tanbal",
                        "tip": "خوشتیپ",
                        "model": "بالا",
                        "color": "یشمی",
                        "pelak": "123ب21",
                        "shasi": "2345",
                        "motor": "ضعیف",
                        "code": "2345678",
                        "vin": "34567876543456"
                    },
                    {
                        "id": 1,
                        "name": "پورشه",
                        "system": "قوی",
                        "tip": "خوشتیپ",
                        "model": "بالا",
                        "color": "مشکی",
                        "pelak": "21ب321",
                        "shasi": "345672",
                        "motor": "قوی",
                        "code": "123",
                        "vin": "tgb12y1212"
                    }
                ],
                "num_of_pages": 1
            }
        """
        parser = RequestParser(request.query_params)
        try:
            name = parser.str("name", None)
            pelak = parser.str("pelak", None)
            username = parser.str("code_melli", None)
            shasi = parser.str("shasi", None)
            datetime = parser.timestamp("datetime", None)
            model = parser.str("model", None)
            system = parser.str("system", None)
        except EmamParameterError as e:
            return raise_error("PARAMETERS_ERROR", e.field_name)

        filter_list = {"is_deleted": False}
        gholname_filter_list = {}

        if pelak:
            filter_list["pelak__contains"] = pelak

        if shasi:
            filter_list["shasi__contains"] = shasi

        if model:
            filter_list["model__contains"] = model

        if system:
            filter_list["system__contains"] = system

        gholnames = []

        if name or username:
            gh = True
            gholnames = Gholname.objects.filter(faskh__isnull=True)
        else:
            gh = False

        if name:
            gholnames = gholnames.filter(
                Q(kharidar__first_name__contains=name) |
                Q(kharidar__last_name__contains=name) |
                Q(foro0shande__first_name__contains=name) |
                Q(foro0shande__last_name__contains=name)
            )

        if username:
            gholnames = gholnames.filter(
                Q(kharidar__username__contains=username) |
                Q(foro0shande__username__contains=username)
            )

        if datetime:
            kharid_foro0shs = KharidForo0shCar.objects.filter(
                gholname_kharid__tahvil_datetime__lte=datetime,
                gholname_foro0sh__tahvil_datetime__gte=datetime
            )
            filter_list["pk__in"] = kharid_foro0shs.values_list("gholname_kharid__car")

        if gh:
            filter_list["id__in"] = Subquery(gholnames.values('car_id'))

        cars = Car.objects.filter(**filter_list).order_by("-created_at")

        paginator = Paginator(cars, NUM_OF_PAGINATION_ITEMS)
        # paginator = Paginator(cars, 1000)
        page = request.query_params.get("page")

        try:
            cars = paginator.page(page)
        except PageNotAnInteger:
            cars = paginator.page(1)
        except EmptyPage:
            cars = []

        serializers = self.serializer_class(cars, many=True).data
        return response({"cars": serializers, "num_of_pages": paginator.num_pages})


class CarCostsView(APIView):
    serializer_class = CarsListInParkingSerializer
    permission_classes = (IsAuthenticated, IsICarAdmin)
    authentication_classes = (TokenAuthentication,)

    def get(self, request, **kwargs):
        """
        daryAfte makhAreje yek mashin
        --------
        parameters
            - name: id
            type: int
            description: id e mashin

        Response Messages
            - code: 200
            response:{
                "total_cost": 400000
                "costs":[
                    {
                        "id": 1,
                        "car": {
                            "id": 1,
                            "name": "پورشه",
                            "system": "kado0tanbal",
                            "tip": "خوشتیپ",
                            "model": "بالا",
                            "color": "یشمی",
                            "pelak": "123ب21",
                            "shasi": "2345",
                            "motor": "ضعیف",
                            "code": "2345678",
                            "vin": "34567876543456",
                            "bime_end_time": "1527038165",
                            "takhfif_bime": 0,
                            "kilometer": 0,
                            "moayene_end_time": "1527038165",
                            "fuel": 1,
                            "color_details": "00000000000"
                        },
                        "created_at": "1527038211",
                        "updated_at": "2018-05-23T09:16:51.008578+04:30",
                        "subject": "تعمیر گلگیر",
                        "amount": 200000,
                        "store": "اسی پنجه طلا",
                        "description": "هزینه سنگینی بود"
                    },
                        {
                        "id": 1,
                        "car": {
                            "id": 1,
                            "name": "پورشه",
                            "system": "kado0tanbal",
                            "tip": "خوشتیپ",
                            "model": "بالا",
                            "color": "یشمی",
                            "pelak": "123ب21",
                            "shasi": "2345",
                            "motor": "ضعیف",
                            "code": "2345678",
                            "vin": "34567876543456",
                            "bime_end_time": "1527038165",
                            "takhfif_bime": 0,
                            "kilometer": 0,
                            "moayene_end_time": "1527038165",
                            "fuel": 1,
                            "color_details": "00000000000"
                        },
                        "created_at": "1527038211",
                        "updated_at": "2018-05-23T09:16:51.008578+04:30",
                        "subject": "تعمیر گلگیر",
                        "amount": 200000,
                        "store": "اسی پنجه طلا",
                        "description": "هزینه سنگینی بود"
                    }
                ]
        """
        car_id = kwargs["id"]

        try:
            car = Car.objects.get(id=car_id)
        except Car.DoesNotExist:
            return raise_error("NOT_FOUND_ERROR", "CAR")

        serializers = ShowCarCostSerializer(car.carcost_set.all(), many=True).data

        total_cost = 0
        for cost in serializers:
            total_cost += int(cost["amount"])

        return response({"costs": serializers, "total_cost": total_cost})

    def delete(self, request, **kwargs):
        """
        hazf e yek kharj az makhArej e mashin
        --------
        parameters
            - name: id
            type: int
            description: id e kharj e morede nazar

        Response Messages
            - code: 200
            response: {"status": "ok"}
        """
        cost_id = kwargs["id"]

        try:
            cost = CarCost.objects.get(id=cost_id)
        except CarCost.DoesNotExist:
            return raise_error("NOT_FOUND_ERROR", "COST")

        cost.delete()

        return response({"status": "ok"})


class CarTypesView(APIView):
    serializer_class = CarTypesListSerializer
    permission_classes = (IsAuthenticated, IsICarAdmin)
    authentication_classes = (TokenAuthentication,)

    def get(self, request):
        """
        daryAfte list e no e mashin ha
        --------

        Response Messages
            - code: 200
            response: [
                {
                    "id": 1,
                    "name": "سواری"
                },
                {
                    "id": 2,
                    "name": "سنگین"
                }
            ]
        """
        car_types = CarType.objects.all()

        serializers = self.serializer_class(car_types, many=True).data

        return response(serializers)


class SystemsView(APIView):
    serializer_class = SystemsListSerializer
    permission_classes = (IsAuthenticated, IsICarAdmin)
    authentication_classes = (TokenAuthentication,)

    def get(self, request, **kwargs):
        """
        daryAfte system haye yek no az mashin
        --------
        parameters
            - name: type_id
            type: int
            description: id e no e mashin

        Response Messages
            - code: 200
            response:[
                {
                    "id": 1,
                    "name": "خوب"
                },
                {
                    "id": 2,
                    "name": "بد"
                }
            ]
        """
        try:
            type_id = kwargs["type_id"]
        except KeyError:
            return response([])
        

        car_systems = CarSystem.objects.filter(car_type__id=type_id)

        serializers = self.serializer_class(car_systems, many=True).data

        return response(serializers)


class TipsView(APIView):
    serializer_class = TipsListSerializer
    permission_classes = (IsAuthenticated, IsICarAdmin)
    authentication_classes = (TokenAuthentication,)

    def get(self, request, **kwargs):
        """
        daryAfte tip haye yek system
        --------
        parameters
            - name: system_id
            type: int
            description: id e system e mashin

        Response Messages
            - code: 200
            response:[
                {
                    "id": 1,
                    "name": "خوشتیپ"
                },
                {
                    "id": 2,
                    "name": "بد تیپ"
                }
            ]
        """
        try:
            system_id = kwargs["system_id"]
        except KeyError:
            return response([])
        
        
        car_tips = CarTip.objects.filter(car_system__id=system_id)

        serializers = self.serializer_class(car_tips, many=True).data

        return response(serializers)


class CarColorsView(APIView):
    serializer_class = CarTypesListSerializer
    permission_classes = (IsAuthenticated, IsICarAdmin)
    authentication_classes = (TokenAuthentication,)

    def get(self, request):
        """
        daryAfte list e rang hAye mashin ha
        --------

        Response Messages
            - code: 200
            response: [
                "سبز",
                "سفید"
            ]
        """
        colors = Color.objects.all().values_list("name", flat=True)

        return response(colors)


class CarSaveView(APIView):
    permission_classes = (IsAuthenticated, IsICarAdmin)
    authentication_classes = (TokenAuthentication,)

    def put(self, request):
        try:
            car = Car.objects.get(pk=request.data['id'])
        except Car.DoesNotExist:
            return raise_error("NOT_FOUND_ERROR", "CAR")

        car.vin = request.data['vin']
        car.code = request.data['code']
        car.save()
        return response({})
