from rest_framework import serializers

from applications.accounting.models import (
    Bank, DaftarKol, HesabDaftarKol, Chek, DaryaftAzTarafHesab,
    PardakhtBeTarafHesab,
    BankTransfer,
    TalabBedeh, TalabBedehGhest)
from applications.users.serializers import UserInGholnameListSerializer


class AllDaftarKol(serializers.ModelSerializer):
    class Meta:
        model = DaftarKol
        fields = '__all__'
        depth = 5


class AllBank(serializers.ModelSerializer):
    class Meta:
        model = Bank
        fields = '__all__'
        depth = 5


class AllHesabDaftarKol(serializers.ModelSerializer):
    class Meta:
        model = HesabDaftarKol
        fields = '__all__'
        depth = 5


class AllChek(serializers.ModelSerializer):
    class Meta:
        model = Chek
        fields = '__all__'
        depth = 5


class AllDaryaftAzTarafHesab(serializers.ModelSerializer):
    class Meta:
        model = DaryaftAzTarafHesab
        fields = '__all__'
        depth = 5


class AllPardakhtBeTarafHesab(serializers.ModelSerializer):
    class Meta:
        model = PardakhtBeTarafHesab
        fields = '__all__'
        depth = 5


class AllTalabBedehGhest(serializers.ModelSerializer):
    class Meta:
        model = TalabBedehGhest
        fields = '__all__'


class AllTalabBedeh(serializers.ModelSerializer):
    talabbedehghest_set = AllTalabBedehGhest(many=True)

    class Meta:
        model = TalabBedeh
        fields = '__all__'
        depth = 5


class AllBankTransfer(serializers.ModelSerializer):
    class Meta:
        model = BankTransfer
        fields = '__all__'
        depth = 5


class BanksListSerializer(serializers.ModelSerializer):
    class Meta:
        model = Bank
        fields = '__all__'


class HesabDaftarKolListSerializer(serializers.ModelSerializer):
    bank = BanksListSerializer()

    class Meta:
        model = HesabDaftarKol
        fields = [
            "id",
            "name",
            "bank",
            "shobe",
            "has_chek",
            "has_card",
            "saheb_hesab",
            "has_pos",
            "code_shobe",
            "shomare_hesab",
            "telephone",
            "tozihat"
        ]


class HesabDaftarKolSerializer(serializers.ModelSerializer):
    class Meta:
        model = HesabDaftarKol
        fields = ["id", "name"]


class DaftarKolListSerializer(serializers.ModelSerializer):
    hesabdaftarkol_set = HesabDaftarKolListSerializer(many=True)

    class Meta:
        model = DaftarKol
        fields = ("id", "name", "hesabdaftarkol_set")


class DaryaftAzTarafHesabListSerializer(serializers.ModelSerializer):
    pardakht_konande = UserInGholnameListSerializer()
    tavassot = UserInGholnameListSerializer()
    hesab_daftar_kol = HesabDaftarKolSerializer()
    created_at = serializers.SerializerMethodField()

    class Meta:
        model = DaryaftAzTarafHesab
        fields = (
            'id',
            'pardakht_konande',
            'tavassot',
            'tarikh_daryaft',
            'jahat',
            'hesab_daftar_kol',
            'bAbat',
            'tozihat',
            'pos_amount',
            'created_at',
            'card_amount',
            'cash'
        )

    def get_created_at(self, pbth):
        return pbth.tarikh_daryaft.strftime('%s')


class PardakhtBeTarafHesabListSerializer(serializers.ModelSerializer):
    daryaft_konande = UserInGholnameListSerializer()
    tavassot = UserInGholnameListSerializer()
    hesab_daftar_kol = HesabDaftarKolSerializer()
    created_at = serializers.SerializerMethodField()

    class Meta:
        model = PardakhtBeTarafHesab
        fields = (
            'id',
            'daryaft_konande',
            'tavassot',
            'tarikh_pardakht',
            'jahat',
            'hesab_daftar_kol',
            'bAbat',
            'tozihat',
            'pos_amount',
            'card_amount',
            'cash',
            'created_at',
            'carmozd'
        )

    def get_created_at(self, pbth):
        return pbth.tarikh_pardakht.strftime('%s')


class BankTransferListSerializer(serializers.ModelSerializer):
    daryaft_konande = serializers.SerializerMethodField()
    tavassot = serializers.SerializerMethodField()
    hesab_daftar_kol = serializers.SerializerMethodField()
    tarikh_pardakht = serializers.SerializerMethodField()
    jahat = serializers.SerializerMethodField()
    bAbat = serializers.SerializerMethodField()
    tozihat = serializers.SerializerMethodField()
    pos_amount = serializers.SerializerMethodField()
    card_amount = serializers.SerializerMethodField()
    cash = serializers.SerializerMethodField()
    created_at = serializers.SerializerMethodField()
    carmozd = serializers.SerializerMethodField()

    class Meta:
        model = BankTransfer
        fields = (
            'id',
            'daryaft_konande',
            'tavassot',
            'tarikh_pardakht',
            'jahat',
            'hesab_daftar_kol',
            'bAbat',
            'tozihat',
            'pos_amount',
            'card_amount',
            'cash',
            'created_at',
            'carmozd'
        )

    def get_created_at(self, bt):
        return bt.amal.strftime('%s')

    def get_hesab_daftar_kol(self, bt):
        """
        :type bt: BankTransfer
        """
        if bt.mabda:
            return HesabDaftarKolSerializer(bt.mabda).data
        return HesabDaftarKolSerializer(bt.maghsad).data

    def get_daryaft_konande(self, bt):
        """
        :type bt: BankTransfer
        """
        if bt.mabda:
            return bt.mabda.bank.name
        return bt.maghsad.bank.name

    def get_tavassot(self, bt):
        """
        :type bt: BankTransfer
        """
        return UserInGholnameListSerializer(bt.tavasot).data

    def get_tarikh_pardakht(self, bt):
        """
        :type bt: BankTransfer
        """
        return bt.amal.strftime('%s')

    def get_jahat(self, bt):
        """
        :type bt: BankTransfer
        """
        return BankTransfer.TARIGHS[bt.tarigh]

    def get_bAbat(self, bt):
        """
        :type bt: BankTransfer
        """
        return bt.babat

    def get_tozihat(self, bt):
        """
        :type bt: BankTransfer
        """
        return bt.description

    def get_pos_amount(self, bt):
        """
        :type bt: BankTransfer
        """
        return 0

    def get_card_amount(self, bt):
        """
        :type bt: BankTransfer
        """
        return 0

    def get_cash(self, bt):
        """
        :type bt: BankTransfer
        """
        return bt.mablagh if bt.mabda else -bt.mablagh

    def get_carmozd(self, bt):
        """
        :type bt: BankTransfer
        """
        return 0


class ChekListSerializer(serializers.ModelSerializer):
    bank = BanksListSerializer()
    daryafti = DaryaftAzTarafHesabListSerializer()
    pardakhti = PardakhtBeTarafHesabListSerializer()
    types = serializers.SerializerMethodField()
    tozihats = serializers.SerializerMethodField()

    class Meta:
        model = Chek
        fields = (
            'id',
            'types',
            'tarikh_sarresid',
            'tarikh_pardakht',
            'bank',
            'shomare_chek',
            'dar_vajh',
            'code',
            'shobe',
            'shomare_hesab',
            'saheb_hesab',
            'mablagh',
            'daryafti',
            'pardakhti',
            'is_self',
            'tozihats'
        )

    def get_types(self, chek):
        """
        :type chek: Chek
        """
        return [
            hasattr(chek, 'chekstatusdelete'),
            hasattr(chek, 'chekstatusvosul'),
            hasattr(chek, 'chekstatusreturn'),
            hasattr(chek, 'chekstatuspass'),
            hasattr(chek, 'chekstatusdelay'),
            False,
            self.context['type']
        ]

    def get_tozihats(self, chek):
        """
        :type chek: Chek
        """
        tozihats = []
        if hasattr(chek, 'chekstatusdelete'):
            tozihats.append(
                chek.chekstatusdelete.tozihat
            )
        if hasattr(chek, 'chekstatusvosul'):
            tozihats.append(
                chek.chekstatusvosul.tozihat
            )
        if hasattr(chek, 'chekstatusreturn'):
            tozihats.append(
                chek.chekstatusreturn.tozihat
            )
        if hasattr(chek, 'chekstatuspass'):
            tozihats.append(
                chek.chekstatuspass.tozihat
            )
        if hasattr(chek, 'chekstatusdelay'):
            tozihats.append(
                chek.chekstatusdelay.tozihat
            )
        return tozihats


class MinimalPardakht(serializers.ModelSerializer):
    amel = serializers.SerializerMethodField()
    # amount = serializers.SerializerMethodField()
    cash_amount = serializers.SerializerMethodField()
    tavassot = UserInGholnameListSerializer()
    sign = serializers.SerializerMethodField()
    tarigh = serializers.SerializerMethodField()
    created_at = serializers.SerializerMethodField()
    chek_amount = serializers.SerializerMethodField()
    bank = serializers.SerializerMethodField()

    class Meta:
        model = PardakhtBeTarafHesab
        fields = (
            'amel',
            'tavassot',
            'tarigh',
            'bank',
            'bAbat',
            'jahat',
            'sign',
            'tozihat',
            'pos_amount',
            'carmozd',
            'cash_amount',
            'card_amount',
            'created_at',
            'chek_amount',
        )

    def get_bank(self, pbht):
        """
        :type pbht: PardakhtBeTarafHesab
        """
        return pbht.hesab_daftar_kol.bank.name

    def get_created_at(self, pbht):
        """
        :type pbht: PardakhtBeTarafHesab
        """
        return pbht.tarikh_pardakht.strftime('%s')

    def get_amel(self, pbht):
        """
        :type pbht: PardakhtBeTarafHesab
        """
        return UserInGholnameListSerializer(pbht.daryaft_konande).data

    def get_sign(self, pbht):
        return True

    def get_cash_amount(self, pbht):
        return 0

    def get_tarigh(self, pbht):
        if pbht.cash > 0:
            return 1
        if pbht.pos_amount > 0:
            return 2
        if pbht.card_amount > 0:
            return 3
        return 0

    def get_chek_amount(self, tb):
        return 0


class MinimalPardakht1(serializers.ModelSerializer):
    amel = serializers.SerializerMethodField()
    # amount = serializers.SerializerMethodField()
    cash_amount = serializers.SerializerMethodField()
    tavassot = UserInGholnameListSerializer()
    sign = serializers.SerializerMethodField()
    tarigh = serializers.SerializerMethodField()
    created_at = serializers.SerializerMethodField()
    chek_amount = serializers.SerializerMethodField()
    sarresid = serializers.SerializerMethodField()
    tpl = serializers.IntegerField(default=0)

    class Meta:
        model = PardakhtBeTarafHesab
        fields = (
            'id',
            'tpl',
            'amel',
            'tavassot',
            'tarigh',
            'bAbat',
            'sign',
            'tozihat',
            'carmozd',
            'pos_amount',
            'cash_amount',
            'card_amount',
            'chek_amount',
            'sarresid',
            'created_at'
        )

    def get_sarresid(self, pbht):
        return []

    def get_created_at(self, pbht):
        """
        :type pbht: PardakhtBeTarafHesab
        """
        return pbht.tarikh_pardakht.strftime('%s')

    def get_amel(self, pbht):
        """
        :type pbht: PardakhtBeTarafHesab
        """
        return UserInGholnameListSerializer(pbht.daryaft_konande).data

    def get_sign(self, pbht):
        return True

    def get_cash_amount(self, pbht):
        return pbht.cash

    def get_tarigh(self, pbht):
        if pbht.cash > 0:
            return 1
        if pbht.pos_amount > 0:
            return 2
        if pbht.card_amount > 0:
            return 3
        return 0

    def get_chek_amount(self, tb):
        return 0


class MinimalDaryaft(serializers.ModelSerializer):
    amel = serializers.SerializerMethodField()
    cash_amount = serializers.SerializerMethodField()
    tavassot = UserInGholnameListSerializer()
    sign = serializers.SerializerMethodField()
    tarigh = serializers.SerializerMethodField()
    created_at = serializers.SerializerMethodField()
    carmozd = serializers.SerializerMethodField()
    chek_amount = serializers.SerializerMethodField()
    bank = serializers.SerializerMethodField()

    class Meta:
        model = DaryaftAzTarafHesab
        fields = (
            'amel',
            'tavassot',
            'tarigh',
            'bAbat',
            'jahat',
            'sign',
            'tozihat',
            'pos_amount',
            'cash_amount',
            'carmozd',
            'card_amount',
            'bank',
            'chek_amount',
            'created_at'
        )

    def get_bank(self, pbht):
        """
        :type pbht: DaryaftAzTarafHesab
        """
        return pbht.hesab_daftar_kol.bank.name

    def get_carmozd(self, p):
        return 0

    def get_created_at(self, pbht):
        """
        :type pbht: DaryaftAzTarafHesab
        """
        return pbht.tarikh_daryaft.strftime('%s')

    def get_amel(self, pbht):
        """
        :type pbht: DaryaftAzTarafHesab
        """
        return UserInGholnameListSerializer(pbht.pardakht_konande).data

    def get_sign(self, pbht):
        return False

    def get_cash_amount(self, pbht):
        return 0

    def get_tarigh(self, pbht):
        if pbht.pos_amount > 0:
            return 2
        if pbht.card_amount > 0:
            return 3
        return 0

    def get_chek_amount(self, tb):
        return 0


class MinimalDaryaft1(serializers.ModelSerializer):
    amel = serializers.SerializerMethodField()
    cash_amount = serializers.SerializerMethodField()
    tavassot = UserInGholnameListSerializer()
    sign = serializers.SerializerMethodField()
    tarigh = serializers.SerializerMethodField()
    created_at = serializers.SerializerMethodField()
    chek_amount = serializers.SerializerMethodField()
    carmozd = serializers.SerializerMethodField()
    sarresid = serializers.SerializerMethodField()
    tpl = serializers.IntegerField(default=1)

    class Meta:
        model = DaryaftAzTarafHesab
        fields = (
            'id',
            'tpl',
            'amel',
            'tavassot',
            'tarigh',
            'bAbat',
            'sign',
            'tozihat',
            'carmozd',
            'pos_amount',
            'cash_amount',
            'sarresid',
            'card_amount',
            'chek_amount',
            'created_at'
        )

    def get_carmozd(self, pbht):
        return 0

    def get_sarresid(self, pbht):
        return []

    def get_created_at(self, pbht):
        """
        :type pbht: DaryaftAzTarafHesab
        """
        return pbht.tarikh_daryaft.strftime('%s')

    def get_amel(self, pbht):
        """
        :type pbht: DaryaftAzTarafHesab
        """
        return UserInGholnameListSerializer(pbht.pardakht_konande).data

    def get_sign(self, pbht):
        return False

    def get_tarigh(self, pbht):
        if pbht.pos_amount > 0:
            return 2
        if pbht.card_amount > 0:
            return 3
        return 0

    def get_cash_amount(self, pbht):
        return pbht.cash

    def get_chek_amount(self, tb):
        return 0


class MinimalBankTransfer(serializers.ModelSerializer):
    amel = serializers.SerializerMethodField()
    pos_amount = serializers.SerializerMethodField()
    cash_amount = serializers.SerializerMethodField()
    card_amount = serializers.SerializerMethodField()
    sign = serializers.SerializerMethodField()
    tarigh = serializers.SerializerMethodField()
    tavassot = serializers.SerializerMethodField()
    bAbat = serializers.SerializerMethodField()
    tozihat = serializers.SerializerMethodField()
    created_at = serializers.SerializerMethodField()
    chek_amount = serializers.SerializerMethodField()
    jahat = serializers.SerializerMethodField()
    carmozd = serializers.SerializerMethodField()
    bank = serializers.SerializerMethodField()

    class Meta:
        model = BankTransfer
        fields = (
            'amel',
            'bank',
            'tavassot',
            'tarigh',
            'chek_amount',
            'bAbat',
            'sign',
            'jahat',
            'tozihat',
            'pos_amount',
            'carmozd',
            'cash_amount',
            'card_amount',
            'created_at'
        )

    def get_carmozd(self, p):
        return p.karmozd

    def get_jahat(self, pbht):
        return 'SANDOGH'

    def get_created_at(self, pbht):
        return pbht.amal.strftime('%s')

    def get_bank(self, pbht):
        """
        :type pbht: BankTransfer
        """
        if pbht.mabda and pbht.maghsad:
            if pbht.mabda.daftar_kol_id == self.context['i']:
                q = pbht.mabda
            else:
                q = pbht.maghsad
        else:
            if pbht.mabda:
                q = pbht.mabda
            else:
                q = pbht.maghsad
        return q.bank.name

    def get_amel(self, pbht):
        """
        :type pbht: BankTransfer
        """
        return UserInGholnameListSerializer(pbht.tavasot).data

    def get_tavassot(self, pbht):
        """
        :type pbht: BankTransfer
        """
        return UserInGholnameListSerializer(pbht.tavasot).data

    def get_bAbat(self, pbht):
        """
        :type pbht: BankTransfer
        """
        return pbht.babat

    def get_pos_amount(self, pbht):
        """
        :type pbht: BankTransfer
        """
        return 0

    def get_cash_amount(self, pbht):
        """
        :type pbht: BankTransfer
        """
        return pbht.mablagh

    def get_card_amount(self, pbht):
        """
        :type pbht: BankTransfer
        """
        return 0

    def get_tozihat(self, pbht):
        """
        :type pbht: BankTransfer
        """
        return pbht.description

    def get_sign(self, pbht):
        """
        :type pbht: BankTransfer
        """
        if pbht.mabda and pbht.maghsad:
            return pbht.mabda.daftar_kol_id == self.context['i']
        return pbht.mabda is not None

    def get_tarigh(self, pbht):
        """
        :type pbht: BankTransfer
        """
        # BankTransfer.
        return 1

    def get_chek_amount(self, tb):
        return 0


class MinimalTalabBedeh(serializers.ModelSerializer):
    amel = serializers.SerializerMethodField()
    tavassot = serializers.SerializerMethodField()
    tarigh = serializers.SerializerMethodField()
    bAbat = serializers.SerializerMethodField()
    sign = serializers.SerializerMethodField()
    pos_amount = serializers.SerializerMethodField()
    cash_amount = serializers.SerializerMethodField()
    card_amount = serializers.SerializerMethodField()
    created_at = serializers.SerializerMethodField()
    chek_amount = serializers.SerializerMethodField()
    sarresid = serializers.SerializerMethodField()
    carmozd = serializers.SerializerMethodField()
    tpl = serializers.IntegerField(default=2)

    class Meta:
        model = TalabBedeh
        fields = (
            'id',
            'tpl',
            'amel',
            'tavassot',
            'tarigh',
            'carmozd',
            'bAbat',
            'sign',
            'tozihat',
            'pos_amount',
            'chek_amount',
            'cash_amount',
            'sarresid',
            'deleted',
            'card_amount',
            'created_at'
        )

    def get_carmozd(self, pbht):
        return 0

    def get_sarresid(self, pbht):
        return []

    def get_created_at(self, tb):
        return tb.tarikh_sabt.strftime('%s')

    def get_amel(self, tb):
        return UserInGholnameListSerializer(tb.shakhs).data

    def get_tavassot(self, tb):
        return None

    def get_tarigh(self, tb):
        return None

    def get_bAbat(self, tb):
        return tb.babat

    def get_sign(self, tb):
        return not tb.is_talab

    def get_chek_amount(self, tb):
        return 0

    def get_pos_amount(self, tb):
        return 0

    def get_cash_amount(self, tb):
        return tb.amount

    def get_card_amount(self, tb):
        return 0


class MinimalTalabBedehGhest(serializers.ModelSerializer):
    amel = serializers.SerializerMethodField()
    tavassot = serializers.SerializerMethodField()
    tarigh = serializers.SerializerMethodField()
    bAbat = serializers.SerializerMethodField()
    sign = serializers.SerializerMethodField()
    tozihat = serializers.SerializerMethodField()
    pos_amount = serializers.SerializerMethodField()
    cash_amount = serializers.SerializerMethodField()
    card_amount = serializers.SerializerMethodField()
    created_at = serializers.SerializerMethodField()
    chek_amount = serializers.SerializerMethodField()
    sarresid = serializers.SerializerMethodField()
    carmozd = serializers.SerializerMethodField()
    id = serializers.SerializerMethodField()
    tpl = serializers.IntegerField(default=4)

    class Meta:
        model = TalabBedehGhest
        fields = (
            'id',
            'amel',
            'tpl',
            'tavassot',
            'tarigh',
            'bAbat',
            'carmozd',
            'sign',
            'tozihat',
            'sarresid',
            'pos_amount',
            'cash_amount',
            'chek_amount',
            'card_amount',
            'deleted',
            'created_at'
        )

    def get_carmozd(self, tb):
        return 0

    def get_id(self, tb):
        return tb.talab_bedeh.id

    def get_sarresid(self, tb):
        if tb.tarikh_amal:
            m = tb.tarikh_amal.strftime('%s')
        else:
            m = None
        return [
            tb.tarikh_daryaft.strftime('%s'),
            m
        ]

    def get_created_at(self, tb):
        return tb.talab_bedeh.tarikh_sabt.strftime('%s')

    def get_amel(self, tb):
        return UserInGholnameListSerializer(tb.talab_bedeh.shakhs).data

    def get_tavassot(self, tb):
        return None

    def get_tarigh(self, tb):
        return None

    def get_chek_amount(self, tb):
        return 0

    def get_bAbat(self, tb):
        return tb.talab_bedeh.babat

    def get_sign(self, tb):
        return not tb.talab_bedeh.is_talab

    def get_tozihat(self, tb):
        return tb.tozihat

    def get_pos_amount(self, tb):
        return 0

    def get_cash_amount(self, tb):
        return tb.amount

    def get_card_amount(self, tb):
        return 0


class MinimalChek(serializers.ModelSerializer):
    amel = serializers.SerializerMethodField()
    tavassot = serializers.SerializerMethodField()
    tarigh = serializers.SerializerMethodField()
    bAbat = serializers.SerializerMethodField()
    tozihat = serializers.SerializerMethodField()
    sign = serializers.SerializerMethodField()
    pos_amount = serializers.SerializerMethodField()
    cash_amount = serializers.SerializerMethodField()
    card_amount = serializers.SerializerMethodField()
    chek_amount = serializers.SerializerMethodField()
    created_at = serializers.SerializerMethodField()
    jahat = serializers.SerializerMethodField()
    carmozd = serializers.SerializerMethodField()
    bank = serializers.SerializerMethodField()
    sarresid = serializers.SerializerMethodField()
    tpl = serializers.IntegerField(default=3)

    class Meta:
        model = Chek
        fields = (
            'id',
            'bank',
            'tpl',
            'amel',
            'tavassot',
            'tarigh',
            'carmozd',
            'bAbat',
            'sign',
            'jahat',
            'sarresid',
            'tozihat',
            'pos_amount',
            'cash_amount',
            'card_amount',
            'chek_amount',
            'created_at'
        )

    def get_carmozd(self, tb):
        return 0

    def get_sarresid(self, tb):
        return []

    def get_tozihat(self, tb):
        """
        :type tb: Chek
        """
        if self.context['i']:
            return tb.chekstatuspass.tozihat
        return tb.chekstatusvosul.tozihat

    def get_bank(self, tb):
        """
        :type tb: Chek
        """
        return tb.bank.name
        # if self.context['i']:
        #     return tb.chekstatuspass.tozihat
        # return tb.chekstatusvosul.tozihat

    def get_created_at(self, tb):
        """
        :type tb: Chek
        """
        if self.context['i']:
            return tb.chekstatuspass.date.strftime('%s')
        return tb.chekstatusvosul.date.strftime('%s')

    def get_amel(self, tb):
        """
        :type tb: Chek
        """
        if tb.daryafti:
            return UserInGholnameListSerializer(tb.daryafti.pardakht_konande).data
        return UserInGholnameListSerializer(tb.pardakhti.daryaft_konande).data

    def get_jahat(self, tb):
        """
        :type tb: Chek
        """
        if tb.daryafti:
            return tb.daryafti.jahat
        return tb.pardakhti.jahat

    def get_tavassot(self, tb):
        """
        :type tb: Chek
        """
        if tb.daryafti:
            return UserInGholnameListSerializer(tb.daryafti.tavassot).data
        return UserInGholnameListSerializer(tb.pardakhti.tavassot).data

    def get_tarigh(self, tb):
        """
        :type tb: Chek
        """
        return None

    def get_bAbat(self, tb):
        """
        :type tb: Chek
        """
        # return tb.daryafti.bAbat if tb.daryafti else tb.pardakhti.bAbat
        return tb.shomare_chek

    def get_sign(self, tb):
        """
        :type tb: Chek
        """
        return not self.context['i']

    def get_pos_amount(self, tb):
        """
        :type tb: Chek
        """
        return 0

    def get_cash_amount(self, tb):
        """
        :type tb: Chek
        """
        return 0

    def get_chek_amount(self, tb):
        """
        :type tb: Chek
        """
        return tb.mablagh

    def get_card_amount(self, tb):
        """
        :type tb: Chek
        """
        return 0


class TalabBedehDaily(serializers.ModelSerializer):
    shakhs = UserInGholnameListSerializer()

    class Meta:
        model = TalabBedeh
        fields = (
            'id',
            'shakhs',
            'babat',
            'amount',
            'tarikh_sabt',
            'tozihat',
            'is_talab',
        )


class TalabBedehGhestDaily(serializers.ModelSerializer):
    talab_bedeh = TalabBedehDaily()

    class Meta:
        model = TalabBedehGhest
        fields = (
            'id',
            'talab_bedeh',
            'num',
            'amount',
            'tarikh_daryaft',
            'tarikh_amal',
        )
