import { InjectionToken, Inject, Injectable, Directive, Input, Output, HostListener, EventEmitter, ElementRef, Renderer2, ChangeDetectorRef, PLATFORM_ID, Pipe, NgModule } from '@angular/core';
import { map } from 'rxjs/operators/map';
import { delay } from 'rxjs/operators/delay';
import { faFacebookF, faTwitter, faLinkedinIn, faGooglePlusG, faPinterestP, faRedditAlien, faTumblr, faWhatsapp, faFacebookMessenger, faTelegramPlane, faVk, faStumbleupon, faXing } from '@fortawesome/free-brands-svg-icons';
import { faCommentAlt, faEnvelope, faCheck, faPrint, faExclamation, faLink } from '@fortawesome/free-solid-svg-icons';
import { BehaviorSubject } from 'rxjs/BehaviorSubject';
import { isPlatformBrowser } from '@angular/common';
import { HttpClient } from '@angular/common/http';
import { catchError } from 'rxjs/operators/catchError';
import { take } from 'rxjs/operators/take';
import { switchMap } from 'rxjs/operators/switchMap';
import { tap } from 'rxjs/operators/tap';
import { filter } from 'rxjs/operators/filter';
import { empty } from 'rxjs/observable/empty';
import { of } from 'rxjs/observable/of';

/**
 * @fileoverview added by tsickle
 * @suppress {checkTypes} checked by tsc
 */
const CONFIG = new InjectionToken('CONFIG');

/**
 * @fileoverview added by tsickle
 * @suppress {checkTypes} checked by tsc
 */
/**
 * Simple object check.
 * @param {?} item
 * @return {?}
 */
function isObject(item) {
    return (item && typeof item === 'object' && !Array.isArray(item));
}
/**
 * Deep merge two objects.
 * @param {?} target
 * @param {...?} sources
 * @return {?}
 */
function mergeDeep(target, ...sources) {
    if (!sources.length) {
        return target;
    }
    const /** @type {?} */ source = sources.shift();
    if (isObject(target) && isObject(source)) {
        for (const /** @type {?} */ key in source) {
            if (isObject(source[key])) {
                if (!target[key]) {
                    Object.assign(target, { [key]: {} });
                }
                mergeDeep(target[key], source[key]);
            }
            else {
                Object.assign(target, { [key]: source[key] });
            }
        }
    }
    return mergeDeep(target, ...sources);
}
/**
 * Returns a readable number from share count
 * @param {?} num
 * @param {?} digits
 * @return {?}
 */
function shareCountFormatter(num, digits) {
    const /** @type {?} */ si = [
        { value: 1E9, symbol: 'B' },
        { value: 1E6, symbol: 'M' },
        { value: 1E3, symbol: 'K' }
    ], /** @type {?} */ rx = /\.0+$|(\.[0-9]*[1-9])0+$/;
    for (let /** @type {?} */ i = 0; i < si.length; i++) {
        if (num >= si[i].value) {
            return (num / si[i].value).toFixed(digits).replace(rx, '$1') + si[i].symbol;
        }
    }
    return num.toFixed(digits).replace(rx, '$1');
}
/**
 * Copy text to clipboard
 * @param {?} text
 * @param {?} ios
 * @return {?}
 */
function copyToClipboard(text, ios) {
    return new Promise((resolve) => {
        // Create a hidden textarea element
        const /** @type {?} */ textArea = document.createElement('textarea');
        textArea.value = text;
        document.body.appendChild(textArea);
        // highlight textarea to copy the text
        if (ios) {
            const /** @type {?} */ range = document.createRange();
            range.selectNodeContents(textArea);
            const /** @type {?} */ selection = window.getSelection();
            selection.removeAllRanges();
            selection.addRange(range);
            textArea.readOnly = true;
            textArea.setSelectionRange(0, 999999);
        }
        else {
            textArea.select();
        }
        document.execCommand('copy');
        document.body.removeChild(textArea);
        resolve();
    });
}
/**
 * Get meta tag content
 * @param {?} key
 * @return {?}
 */
function getMetaContent(key) {
    const /** @type {?} */ metaTag = document.querySelector(`meta[property="${key}"]`);
    return metaTag ? metaTag.getAttribute('content') : undefined;
}
/**
 * Detect operating system 'ios', 'android', or 'desktop'
 * @return {?}
 */
function getOS() {
    const /** @type {?} */ userAgent = navigator.userAgent || navigator.vendor || (/** @type {?} */ (window)).opera;
    if (/android/i.test(userAgent)) {
        return 'android';
    }
    if (/iPad|iPhone|iPod/.test(userAgent) && !(/** @type {?} */ (window)).MSStream) {
        return 'ios';
    }
    return 'desktop';
}
/**
 * Returns a valid URL or falls back to current URL
 * @param {?} url
 * @param {?} fallbackUrl
 * @return {?}
 */
function getValidUrl(url, fallbackUrl) {
    if (url) {
        const /** @type {?} */ r = /(http|https):\/\/(\w+:{0,1}\w*@)?(\S+)(:[0-9]+)?(\/|\/([\w#!:.?+=&%@!\-\/]))?/;
        if (r.test(url)) {
            return url;
        }
        console.warn(`[ShareButtons]: Sharing link '${url}' is invalid!`);
    }
    return fallbackUrl;
}

/**
 * @fileoverview added by tsickle
 * @suppress {checkTypes} checked by tsc
 */
/**
 * Meta tags operator - Serialize meta tags into the sharer URL
 */
const metaTagsOperators = [
    map((ref) => {
        // Social network sharer URL */
        const /** @type {?} */ SharerURL = ref.prop.share[ref.os];
        if (SharerURL) {
            // object contains supported meta tags
            const /** @type {?} */ metaTags = ref.prop.share.metaTags;
            // object contains meta tags values */
            const /** @type {?} */ metaTagsValues = ref.metaTags;
            let /** @type {?} */ link = '';
            // Set each meta tag with user value
            if (metaTags) {
                link = Object.entries(metaTags).map(([key, metaTag]) => metaTagsValues[key] ? `${metaTag}=${encodeURIComponent(metaTagsValues[key])}` : '').join('&');
            }
            return SharerURL + link;
        }
        return;
    })
];
/**
 * Print button operator
 */
const printOperators = [
    map(() => window.print())
];
/**
 * Copy link to clipboard, used for copy button
 */
const copyOperators = [
    map((ref) => {
        // Disable the button
        ref.renderer.setStyle(ref.el, 'pointer-events', 'none');
        ref.temp = { text: ref.prop.text, icon: ref.prop.icon };
        const /** @type {?} */ link = decodeURIComponent(ref.metaTags.url);
        copyToClipboard(link, ref.os === 'ios')
            .then(() => {
            ref.prop.text = ref.prop.successText;
            ref.prop.icon = ref.prop.successIcon;
        })
            .catch(() => {
            ref.prop.text = ref.prop.failText;
            ref.prop.icon = ref.prop.failIcon;
        })
            .then(() => ref.cd.markForCheck());
        return ref;
    }),
    delay(2000),
    map((ref) => {
        // Enable the button
        ref.renderer.setStyle(ref.el, 'pointer-events', 'auto');
        // Reset copy button text and icon */
        ref.prop.text = ref.temp.text;
        ref.prop.icon = ref.temp.icon;
        ref.cd.markForCheck();
    })
];
/**
 * Add the share URL to message body, used for WhatsApp and Email buttons
 */
const urlInMessageOperators = [
    map((ref) => {
        const /** @type {?} */ description = ref.metaTags.description;
        const /** @type {?} */ url = ref.metaTags.url;
        const /** @type {?} */ newRef = {
            metaTags: {
                description: description ? `${description}\r\n${url}` : url
            }
        };
        return mergeDeep(ref, newRef);
    })
];
const FacebookCountOperators = [
    map((res) => +res.share.share_count)
];
const PinterestCountOperators = [
    map((text) => JSON.parse(text.replace(/^receiveCount\((.*)\)/, '$1'))),
    map((res) => +res.count)
];
const TumblrCountOperators = [
    map((res) => +res.response.note_count)
];
const RedditCountOperators = [
    map((res) => +res.data.children[0].data.score)
];

/**
 * @fileoverview added by tsickle
 * @suppress {checkTypes} checked by tsc
 */
const shareButtonsProp = {
    facebook: {
        type: 'facebook',
        text: 'Facebook',
        icon: faFacebookF,
        color: '#4267B2',
        share: {
            desktop: 'https://www.facebook.com/sharer/sharer.php?',
            android: 'https://www.facebook.com/sharer/sharer.php?',
            ios: 'https://www.facebook.com/sharer/sharer.php?',
            operators: metaTagsOperators,
            metaTags: {
                url: 'u'
            }
        },
        count: {
            request: 'http',
            url: 'https://graph.facebook.com?id=',
            operators: FacebookCountOperators
        }
    },
    twitter: {
        type: 'twitter',
        text: 'Twitter',
        icon: faTwitter,
        color: '#00acee',
        share: {
            desktop: 'https://twitter.com/intent/tweet?',
            android: 'https://twitter.com/intent/tweet?',
            ios: 'https://twitter.com/intent/tweet?',
            operators: metaTagsOperators,
            metaTags: {
                url: 'url',
                description: 'text',
                tags: 'hashtags',
                via: 'via'
            }
        }
    },
    google: {
        type: 'google',
        text: 'Google+',
        icon: faGooglePlusG,
        color: '#DB4437',
        share: {
            desktop: 'https://plus.google.com/share?',
            android: 'https://plus.google.com/share?',
            ios: 'https://plus.google.com/share?',
            metaTags: {
                url: 'url',
            },
            operators: metaTagsOperators
        }
    },
    linkedin: {
        type: 'linkedin',
        text: 'LinkedIn',
        icon: faLinkedinIn,
        color: '#006fa6',
        share: {
            desktop: 'http://www.linkedin.com/shareArticle?',
            android: 'http://www.linkedin.com/shareArticle?',
            ios: 'http://www.linkedin.com/shareArticle?',
            operators: metaTagsOperators,
            metaTags: {
                url: 'url',
                title: 'title',
                description: 'summary'
            },
        }
    },
    pinterest: {
        type: 'pinterest',
        text: 'Pinterest',
        icon: faPinterestP,
        color: '#BD091D',
        share: {
            desktop: 'https://in.pinterest.com/pin/create/button/?',
            android: 'https://in.pinterest.com/pin/create/button/?',
            ios: 'https://in.pinterest.com/pin/create/button/?',
            operators: metaTagsOperators,
            metaTags: {
                url: 'url',
                description: 'description',
                image: 'media'
            }
        },
        count: {
            request: 'http',
            url: 'https://api.pinterest.com/v1/urls/count.json?url=',
            args: { responseType: 'text' },
            operators: PinterestCountOperators
        }
    },
    reddit: {
        type: 'reddit',
        text: 'Reddit',
        icon: faRedditAlien,
        color: '#FF4006',
        share: {
            desktop: 'http://www.reddit.com/submit?',
            android: 'http://www.reddit.com/submit?',
            ios: 'http://www.reddit.com/submit?',
            operators: metaTagsOperators,
            metaTags: {
                url: 'url',
                title: 'title'
            },
        },
        count: {
            request: 'http',
            url: 'https://buttons.reddit.com/button_info.json?url=',
            operators: RedditCountOperators
        },
    },
    tumblr: {
        type: 'tumblr',
        text: 'Tumblr',
        icon: faTumblr,
        color: '#36465D',
        share: {
            desktop: 'http://tumblr.com/widgets/share/tool?',
            android: 'http://tumblr.com/widgets/share/tool?',
            ios: 'http://tumblr.com/widgets/share/tool?',
            operators: metaTagsOperators,
            metaTags: {
                url: 'canonicalUrl',
                description: 'caption',
                tags: 'tags'
            }
        },
        count: {
            request: 'jsonp',
            url: 'https://api.tumblr.com/v2/share/stats?url=',
            operators: TumblrCountOperators
        }
    },
    whatsapp: {
        type: 'whatsapp',
        text: 'WhatsApp',
        icon: faWhatsapp,
        color: '#25D366',
        share: {
            desktop: 'https://web.whatsapp.com/send?',
            android: 'whatsapp://send?',
            ios: 'whatsapp://send?',
            operators: [
                ...urlInMessageOperators,
                ...metaTagsOperators
            ],
            metaTags: {
                description: 'text'
            }
        }
    },
    messenger: {
        type: 'messenger',
        text: 'Messenger',
        icon: faFacebookMessenger,
        color: '#0080FF',
        share: {
            android: 'fb-messenger://share/?',
            ios: 'fb-messenger://share/?',
            operators: metaTagsOperators,
            metaTags: {
                url: 'link'
            }
        }
    },
    telegram: {
        type: 'telegram',
        text: 'Telegram',
        icon: faTelegramPlane,
        color: '#0088cc',
        share: {
            desktop: 'https://t.me/share/url?',
            android: 'https://t.me/share/url?',
            ios: 'https://t.me/share/url?',
            operators: metaTagsOperators,
            metaTags: {
                url: 'url',
                description: 'text'
            }
        }
    },
    vk: {
        type: 'vk',
        text: 'VKontakte',
        icon: faVk,
        color: '#4C75A3',
        share: {
            desktop: 'http://vk.com/share.php?',
            android: 'http://vk.com/share.php?',
            ios: 'http://vk.com/share.php?',
            operators: metaTagsOperators,
            metaTags: {
                url: 'url'
            }
        }
    },
    stumble: {
        type: 'stumble',
        text: 'Stumble',
        icon: faStumbleupon,
        color: '#eb4924',
        share: {
            desktop: 'http://www.stumbleupon.com/submit?',
            android: 'http://www.stumbleupon.com/submit?',
            ios: 'http://www.stumbleupon.com/submit?',
            operators: metaTagsOperators,
            metaTags: {
                url: 'url'
            }
        }
    },
    xing: {
        type: 'xing',
        text: 'Xing',
        icon: faXing,
        color: '#006567',
        share: {
            desktop: 'https://www.xing.com/app/user?op=share&',
            android: 'https://www.xing.com/app/user?op=share&',
            ios: 'https://www.xing.com/app/user?op=share&',
            operators: metaTagsOperators,
            metaTags: {
                url: 'url'
            }
        }
    },
    sms: {
        type: 'sms',
        text: 'SMS',
        icon: faCommentAlt,
        color: '#20c16c',
        share: {
            desktop: 'sms:?',
            android: 'sms:?',
            ios: 'sms:?',
            metaTags: {
                description: 'body'
            },
            operators: [
                ...urlInMessageOperators,
                ...metaTagsOperators
            ]
        }
    },
    email: {
        type: 'email',
        text: 'Email',
        icon: faEnvelope,
        color: '#FF961C',
        share: {
            desktop: 'mailto:?',
            android: 'mailto:?',
            ios: 'mailto:?',
            operators: [
                ...urlInMessageOperators,
                ...metaTagsOperators
            ],
            metaTags: {
                title: 'subject',
                description: 'body'
            }
        }
    },
    copy: {
        type: 'copy',
        text: 'Copy link',
        successText: 'Copied',
        successIcon: faCheck,
        failText: 'Error',
        failIcon: faExclamation,
        icon: faLink,
        color: '#607D8B',
        share: {
            operators: copyOperators
        }
    },
    print: {
        type: 'print',
        text: 'Print',
        icon: faPrint,
        color: '#765AA2',
        share: {
            operators: printOperators
        }
    }
};

/**
 * @fileoverview added by tsickle
 * @suppress {checkTypes} checked by tsc
 */
class ShareButtons {
    /**
     * @param {?} config
     */
    constructor(config) {
        this.config = {
            prop: shareButtonsProp,
            options: {
                theme: 'default',
                include: [],
                exclude: [],
                size: 0,
                url: undefined,
                title: undefined,
                description: undefined,
                image: undefined,
                tags: undefined,
                twitterAccount: undefined,
                autoSetMeta: true,
                gaTracking: false,
                windowWidth: 800,
                windowHeight: 500
            }
        };
        this.config$ = new BehaviorSubject(this.config);
        if (config) {
            this.setConfig(config);
        }
    }
    /**
     * @return {?}
     */
    get prop() {
        return this.config.prop;
    }
    /**
     * @return {?}
     */
    get twitterAccount() {
        return this.config.options.twitterAccount;
    }
    /**
     * @return {?}
     */
    get theme() {
        return this.config.options.theme;
    }
    /**
     * @return {?}
     */
    get windowSize() {
        return `width=${this.config.options.windowWidth}, height=${this.config.options.windowHeight}`;
    }
    /**
     * @return {?}
     */
    get url() {
        return this.config.options.url;
    }
    /**
     * @return {?}
     */
    get title() {
        return this.config.options.title;
    }
    /**
     * @return {?}
     */
    get description() {
        return this.config.options.description;
    }
    /**
     * @return {?}
     */
    get image() {
        return this.config.options.image;
    }
    /**
     * @return {?}
     */
    get tags() {
        return this.config.options.tags;
    }
    /**
     * @return {?}
     */
    get autoSetMeta() {
        return this.config.options.autoSetMeta;
    }
    /**
     * @return {?}
     */
    get gaTracking() {
        return this.config.options.gaTracking;
    }
    /**
     * @return {?}
     */
    get size() {
        return this.config.options.size;
    }
    /**
     * @param {?} config
     * @return {?}
     */
    setConfig(config) {
        this.config = mergeDeep(this.config, config);
        this.config$.next(this.config);
    }
    /**
     * @param {?} name
     * @param {?} data
     * @return {?}
     */
    addButton(name, data) {
        const /** @type {?} */ config = {
            prop: Object.assign({}, shareButtonsProp, { [name]: data })
        };
        this.setConfig(config);
    }
}
ShareButtons.decorators = [
    { type: Injectable },
];
/** @nocollapse */
ShareButtons.ctorParameters = () => [
    { type: undefined, decorators: [{ type: Inject, args: [CONFIG,] },] },
];

/**
 * @fileoverview added by tsickle
 * @suppress {checkTypes} checked by tsc
 */
class ShareButtonDirective {
    /**
     * @param {?} shareService
     * @param {?} http
     * @param {?} renderer
     * @param {?} cd
     * @param {?} el
     * @param {?} platform
     */
    constructor(shareService, http, renderer, cd, el, platform) {
        this.shareService = shareService;
        this.http = http;
        this.renderer = renderer;
        this.cd = cd;
        this.el = el;
        this.platform = platform;
        /**
         * Get share count
         */
        this.getCount = false;
        /**
         * Set meta tags from document head, useful when SEO is supported
         */
        this.autoSetMeta = this.shareService.autoSetMeta;
        /**
         * Meta tags inputs - initialized from the global options
         */
        this.url = this.shareService.url;
        this.title = this.shareService.title;
        this.description = this.shareService.description;
        this.image = this.shareService.image;
        this.tags = this.shareService.tags;
        /**
         * Stream that emits when share count is fetched
         */
        this.count = new EventEmitter();
        /**
         * Stream that emits when share dialog is opened
         */
        this.opened = new EventEmitter();
        /**
         * Stream that emits when share dialog is closed
         */
        this.closed = new EventEmitter();
    }
    /**
     * Share link on element click
     * @return {?}
     */
    onClick() {
        if (isPlatformBrowser(this.platform)) {
            const /** @type {?} */ metaTags = this.autoSetMeta ? {
                url: this.url,
                title: this.title || getMetaContent('og:title'),
                description: this.description || getMetaContent('og:description'),
                image: this.image || getMetaContent('og:image'),
                via: this.shareService.twitterAccount || getMetaContent('twitter:site'),
                tags: this.tags,
            } : {
                url: this.url,
                title: this.title,
                description: this.description,
                image: this.image,
                tags: this.tags,
                via: this.shareService.twitterAccount,
            };
            const /** @type {?} */ ref = {
                cd: this.cd,
                renderer: this.renderer,
                prop: this.prop,
                el: this.el.nativeElement,
                os: getOS(),
                metaTags
            };
            // Share the link
            of(ref).pipe(...this.prop.share.operators, tap((sharerURL) => this.share(sharerURL)), take(1)).subscribe();
        }
    }
    /**
     * @param {?} changes
     * @return {?}
     */
    ngOnChanges(changes) {
        if (isPlatformBrowser(this.platform)) {
            if (changes['shareButton'] && (changes['shareButton'].firstChange || changes['shareButton'].previousValue !== this.shareButton)) {
                this.createShareButton(this.shareButton);
            }
            if (!this.url || (changes['url'] && changes['url'].previousValue !== this.url)) {
                of(null).pipe(map(() => {
                    this.url = getValidUrl(this.autoSetMeta ? this.url || getMetaContent('og:url') : this.url, window.location.href);
                    return this.url;
                }), filter(() => this.prop.count && this.getCount), switchMap((url) => this.shareCount(url)), tap((shareCount) => this.count.emit(shareCount)), take(1)).subscribe();
            }
        }
    }
    /**
     * Open sharing dialog
     * @param {?} url - Share URL
     * @return {?}
     */
    share(url) {
        if (url) {
            // GA Tracking
            if (this.shareService.gaTracking && typeof ga !== 'undefined') {
                ga('send', 'social', this.prop.type, 'click', this.url);
            }
            // Emit when share dialog is opened
            this.opened.emit(this.prop.type);
            const /** @type {?} */ popUp = window.open(url, 'newwindow', this.shareService.windowSize);
            // Emit when share dialog is closed
            if (popUp) {
                const /** @type {?} */ pollTimer = window.setInterval(() => {
                    if (popUp.closed) {
                        window.clearInterval(pollTimer);
                        this.closed.emit(this.prop.type);
                    }
                }, 200);
            }
        }
    }
    /**
     * Get link share count
     * @param {?} url - Share URL
     * @return {?} Share count
     */
    shareCount(url) {
        if (this.prop.count.request === 'jsonp') {
            return this.http.jsonp(this.prop.count.url + url, 'callback').pipe(...this.prop.count.operators, catchError(() => empty()));
        }
        else {
            return this.http.get(this.prop.count.url + url, this.prop.count.args).pipe(...this.prop.count.operators, catchError(() => empty()));
        }
    }
    /**
     * @param {?} buttonsName
     * @return {?}
     */
    createShareButton(buttonsName) {
        const /** @type {?} */ button = Object.assign({}, this.shareService.prop[buttonsName]);
        if (button) {
            // Set share button properties
            this.prop = button;
            // Remove previous button class
            this.renderer.removeClass(this.el.nativeElement, `sb-${this.buttonClass}`);
            // Add new button class
            this.renderer.addClass(this.el.nativeElement, `sb-${button.type}`);
            // Set button css color variable
            this.el.nativeElement.style.setProperty('--button-color', this.prop.color);
            // Keep a copy of the class for future replacement
            this.buttonClass = button.type;
        }
        else {
            throw new Error(`[ShareButtons]: The share button '${buttonsName}' does not exist!`);
        }
    }
}
ShareButtonDirective.decorators = [
    { type: Directive, args: [{
                selector: '[shareButton]'
            },] },
];
/** @nocollapse */
ShareButtonDirective.ctorParameters = () => [
    { type: ShareButtons, },
    { type: HttpClient, },
    { type: Renderer2, },
    { type: ChangeDetectorRef, },
    { type: ElementRef, },
    { type: Object, decorators: [{ type: Inject, args: [PLATFORM_ID,] },] },
];
ShareButtonDirective.propDecorators = {
    "shareButton": [{ type: Input },],
    "getCount": [{ type: Input },],
    "autoSetMeta": [{ type: Input },],
    "url": [{ type: Input },],
    "title": [{ type: Input },],
    "description": [{ type: Input },],
    "image": [{ type: Input },],
    "tags": [{ type: Input },],
    "count": [{ type: Output },],
    "opened": [{ type: Output },],
    "closed": [{ type: Output },],
    "onClick": [{ type: HostListener, args: ['click',] },],
};

/**
 * @fileoverview added by tsickle
 * @suppress {checkTypes} checked by tsc
 */
class ShareCountPipe {
    /**
     * @param {?} num
     * @param {?=} digits
     * @return {?}
     */
    transform(num, digits) {
        return shareCountFormatter(num, digits);
    }
}
ShareCountPipe.decorators = [
    { type: Pipe, args: [{
                name: 'shareCount'
            },] },
];
/** @nocollapse */
ShareCountPipe.ctorParameters = () => [];

/**
 * @fileoverview added by tsickle
 * @suppress {checkTypes} checked by tsc
 */
/**
 * @param {?} config
 * @return {?}
 */
function ShareButtonsFactory(config) {
    return new ShareButtons(config);
}
class ShareModule {
    /**
     * @param {?=} config
     * @return {?}
     */
    static forRoot(config) {
        return {
            ngModule: ShareModule,
            providers: [
                { provide: CONFIG, useValue: config },
                {
                    provide: ShareButtons,
                    useFactory: ShareButtonsFactory,
                    deps: [CONFIG]
                }
            ]
        };
    }
}
ShareModule.decorators = [
    { type: NgModule, args: [{
                declarations: [
                    ShareButtonDirective,
                    ShareCountPipe
                ],
                exports: [
                    ShareButtonDirective,
                    ShareCountPipe
                ]
            },] },
];
/** @nocollapse */
ShareModule.ctorParameters = () => [];

/**
 * @fileoverview added by tsickle
 * @suppress {checkTypes} checked by tsc
 */

/**
 * @fileoverview added by tsickle
 * @suppress {checkTypes} checked by tsc
 */
/**
 * Generated bundle index. Do not edit.
 */

export { ShareButtonsFactory, ShareModule, CONFIG, ShareButtons, ShareButtonDirective, ShareCountPipe, shareButtonsProp, metaTagsOperators, printOperators, copyOperators, urlInMessageOperators, FacebookCountOperators, PinterestCountOperators, TumblrCountOperators, RedditCountOperators };
//# sourceMappingURL=ngx-share-core.js.map
