import {
  Component,
  ElementRef,
  ViewEncapsulation,
  EventEmitter,
  SimpleChanges,
  Output,
  NgZone,
  AfterViewInit, AfterViewChecked, OnChanges, OnDestroy
} from '@angular/core';

import { OptionBuilder } from '../services/option-builder';
import { NavigatorGeolocation } from '../services/navigator-geolocation';
import { GeoCoder } from '../services/geo-coder';
import { NguiMap } from '../services/ngui-map';
import { NgMapApiLoader } from '../services/api-loader';
import { InfoWindow } from './info-window';
import { Subject } from 'rxjs/Subject';
import { debounceTime } from 'rxjs/operator/debounceTime';
import { toCamelCase } from '../services/util';

const INPUTS = [
  'backgroundColor', 'center', 'disableDefaultUI', 'disableDoubleClickZoom', 'draggable', 'draggableCursor',
  'draggingCursor', 'heading', 'keyboardShortcuts', 'mapMaker', 'mapTypeControl', 'mapTypeId', 'maxZoom', 'minZoom',
  'noClear', 'overviewMapControl', 'panControl', 'panControlOptions', 'rotateControl', 'scaleControl', 'scrollwheel',
  'streetView', 'styles', 'tilt', 'zoom', 'streetViewControl', 'zoomControl', 'zoomControlOptions', 'mapTypeControlOptions',
  'overviewMapControlOptions', 'rotateControlOptions', 'scaleControlOptions', 'streetViewControlOptions', 'fullscreenControl', 'fullscreenControlOptions',
  'options',
  // ngui-map-specific inputs
  'geoFallbackCenter'
];

const OUTPUTS = [
  'bounds_changed', 'center_changed', 'click', 'dblclick', 'drag', 'dragend', 'dragstart', 'heading_changed', 'idle',
  'typeid_changed', 'mousemove', 'mouseout', 'mouseover', 'projection_changed', 'resize', 'rightclick',
  'tilesloaded', 'tile_changed', 'zoom_changed',
  // to avoid DOM event conflicts
  'mapClick', 'mapMouseover', 'mapMouseout', 'mapMousemove', 'mapDrag', 'mapDragend', 'mapDragstart'
];

@Component({
  selector: 'ngui-map',
  providers: [NguiMap, OptionBuilder, GeoCoder, NavigatorGeolocation],
  styles: [`
    ngui-map {display: block; height: 300px;}
    .google-map {width: 100%; height: 100%}
  `],
  inputs: INPUTS,
  outputs: OUTPUTS,
  encapsulation: ViewEncapsulation.None,
  template: `
    <div class="google-map"></div>
    <ng-content></ng-content>
  `,
})
export class NguiMapComponent implements OnChanges, OnDestroy, AfterViewInit, AfterViewChecked {
  // declare INPUTS for AOT compiler
  public backgroundColor: any; // generated by ngc-pre-compiler.js
  public center: any; // generated by ngc-pre-compiler.js
  public disableDefaultUI: any; // generated by ngc-pre-compiler.js
  public disableDoubleClickZoom: any; // generated by ngc-pre-compiler.js
  public draggable: any; // generated by ngc-pre-compiler.js
  public draggableCursor: any; // generated by ngc-pre-compiler.js
  public draggingCursor: any; // generated by ngc-pre-compiler.js
  public heading: any; // generated by ngc-pre-compiler.js
  public keyboardShortcuts: any; // generated by ngc-pre-compiler.js
  public mapMaker: any; // generated by ngc-pre-compiler.js
  public mapTypeControl: any; // generated by ngc-pre-compiler.js
  public mapTypeId: any; // generated by ngc-pre-compiler.js
  public maxZoom: any; // generated by ngc-pre-compiler.js
  public minZoom: any; // generated by ngc-pre-compiler.js
  public noClear: any; // generated by ngc-pre-compiler.js
  public overviewMapControl: any; // generated by ngc-pre-compiler.js
  public panControl: any; // generated by ngc-pre-compiler.js
  public panControlOptions: any; // generated by ngc-pre-compiler.js
  public rotateControl: any; // generated by ngc-pre-compiler.js
  public scaleControl: any; // generated by ngc-pre-compiler.js
  public scrollwheel: any; // generated by ngc-pre-compiler.js
  public streetView: any; // generated by ngc-pre-compiler.js
  public styles: any; // generated by ngc-pre-compiler.js
  public tilt: any; // generated by ngc-pre-compiler.js
  public zoom: any; // generated by ngc-pre-compiler.js
  public streetViewControl: any; // generated by ngc-pre-compiler.js
  public zoomControl: any; // generated by ngc-pre-compiler.js
  public zoomControlOptions: any; // generated by ngc-pre-compiler.js
  public mapTypeControlOptions: any; // generated by ngc-pre-compiler.js
  public overviewMapControlOptions: any; // generated by ngc-pre-compiler.js
  public rotateControlOptions: any; // generated by ngc-pre-compiler.js
  public scaleControlOptions: any; // generated by ngc-pre-compiler.js
  public streetViewControlOptions: any; // generated by ngc-pre-compiler.js
  public fullscreenControl: any; // generated by ngc-pre-compiler.js
  public fullscreenControlOptions: any; // generated by ngc-pre-compiler.js
  public options: any; // generated by ngc-pre-compiler.js
  public geoFallbackCenter: any; // generated by ngc-pre-compiler.js

  // declare OUTPUTS for AOT compiler
  public bounds_changed: any; // generated by ngc-pre-compiler.js
  public center_changed: any; // generated by ngc-pre-compiler.js
  public click: any; // generated by ngc-pre-compiler.js
  public dblclick: any; // generated by ngc-pre-compiler.js
  public drag: any; // generated by ngc-pre-compiler.js
  public dragend: any; // generated by ngc-pre-compiler.js
  public dragstart: any; // generated by ngc-pre-compiler.js
  public heading_changed: any; // generated by ngc-pre-compiler.js
  public idle: any; // generated by ngc-pre-compiler.js
  public typeid_changed: any; // generated by ngc-pre-compiler.js
  public mousemove: any; // generated by ngc-pre-compiler.js
  public mouseout: any; // generated by ngc-pre-compiler.js
  public mouseover: any; // generated by ngc-pre-compiler.js
  public projection_changed: any; // generated by ngc-pre-compiler.js
  public resize: any; // generated by ngc-pre-compiler.js
  public rightclick: any; // generated by ngc-pre-compiler.js
  public tilesloaded: any; // generated by ngc-pre-compiler.js
  public tile_changed: any; // generated by ngc-pre-compiler.js
  public zoom_changed: any; // generated by ngc-pre-compiler.js
  public mapClick: any; // generated by ngc-pre-compiler.js
  public mapMouseover: any; // generated by ngc-pre-compiler.js
  public mapMouseout: any; // generated by ngc-pre-compiler.js
  public mapMousemove: any; // generated by ngc-pre-compiler.js
  public mapDrag: any; // generated by ngc-pre-compiler.js
  public mapDragend: any; // generated by ngc-pre-compiler.js
  public mapDragstart: any; // generated by ngc-pre-compiler.js

  @Output() public mapReady$: EventEmitter<any> = new EventEmitter();

  public el: HTMLElement;
  public map: google.maps.Map;
  public mapOptions: google.maps.MapOptions = {};

  public inputChanges$ = new Subject();

  // map objects by group
  public infoWindows: { [id: string]: InfoWindow } = { };

  // map has been fully initialized
  public mapIdledOnce: boolean = false;

  private initializeMapAfterDisplayed = false;
  private apiLoaderSub;

  constructor(
    public optionBuilder: OptionBuilder,
    public elementRef: ElementRef,
    public geolocation: NavigatorGeolocation,
    public geoCoder: GeoCoder,
    public nguiMap: NguiMap,
    public apiLoader: NgMapApiLoader,
    public zone: NgZone,
  ) {
    apiLoader.load();

    // all outputs needs to be initialized,
    // http://stackoverflow.com/questions/37765519/angular2-directive-cannot-read-property-subscribe-of-undefined-with-outputs
    OUTPUTS.forEach(output => this[output] = new EventEmitter());
  }

  ngAfterViewInit() {
    this.apiLoaderSub = this.apiLoader.api$.subscribe(() => this.initializeMap());
  }

  ngAfterViewChecked() {
      if (this.initializeMapAfterDisplayed && this.el && this.el.offsetWidth > 0) {
        this.initializeMap();
      }
  }

  ngOnChanges(changes: SimpleChanges) {
    this.inputChanges$.next(changes);
  }

  initializeMap(): void {
    this.el = this.elementRef.nativeElement.querySelector('.google-map');
    if (this.el && this.el.offsetWidth === 0) {
        this.initializeMapAfterDisplayed = true;
        return;
    }

    this.initializeMapAfterDisplayed = false;
    this.mapOptions = this.optionBuilder.googlizeAllInputs(INPUTS, this);


    this.mapOptions.zoom = this.mapOptions.zoom || 15;
    typeof this.mapOptions.center === 'string' && (delete this.mapOptions.center);

    this.zone.runOutsideAngular(() => {
      this.map = new google.maps.Map(this.el, this.mapOptions);
      this.map['mapObjectName'] = 'NguiMapComponent';

      if (!this.mapOptions.center) { // if center is not given as lat/lng
        this.setCenter();
      }

      // set google events listeners and emits to this outputs listeners
      this.nguiMap.setObjectEvents(OUTPUTS, this, 'map');

      this.map.addListener('idle', () => {
        if (!this.mapIdledOnce) {
          this.mapIdledOnce = true;
          setTimeout(() => { // Why????, subsribe and emit must not be in the same cycle???
            this.mapReady$.emit(this.map);
          });
        }
      });

      // update map when input changes
      debounceTime.call(this.inputChanges$, 1000)
        .subscribe((changes: SimpleChanges) => this.nguiMap.updateGoogleObject(this.map, changes));

      if (typeof window !== 'undefined' && (<any>window)['nguiMapRef']) {
        // expose map object for test and debugging on (<any>window)
        (<any>window)['nguiMapRef'].map = this.map;
      }
    });
  }

  setCenter(): void {
    if (!this['center']) { // center is not from user. Thus, we set the current location
      this.geolocation.getCurrentPosition().subscribe(
        position => {

          let latLng = new google.maps.LatLng(position.coords.latitude, position.coords.longitude);
          this.map.setCenter(latLng);
        },
        error => {
          console.error('ngui-map: Error finding the current position');
          this.map.setCenter(this.mapOptions['geoFallbackCenter'] || new google.maps.LatLng(0, 0));
        }
      );
    }
    else if (typeof this['center'] === 'string') {
      this.geoCoder.geocode({address: this['center']}).subscribe(
        results => {

          this.map.setCenter(results[0].geometry.location);
        },
        error => {
          this.map.setCenter(this.mapOptions['geoFallbackCenter'] || new google.maps.LatLng(0, 0));
        });
    }
  }

  openInfoWindow(id: string, anchor: google.maps.MVCObject) {
    this.infoWindows[id].open(anchor);
  }

  closeInfoWindow(id: string) {
    // if infoWindow for id exists, close the infoWindow
    if (this.infoWindows[id])
      this.infoWindows[id].close();
  }

  ngOnDestroy() {
    this.inputChanges$.complete();
    if (this.el && !this.initializeMapAfterDisplayed) {
      this.nguiMap.clearObjectEvents(OUTPUTS, this, 'map');
    }
    if (this.apiLoaderSub) {
      this.apiLoaderSub.unsubscribe();
    }
  }

  // map.markers, map.circles, map.heatmapLayers.. etc
  addToMapObjectGroup(mapObjectName: string, mapObject: any) {
    let groupName = toCamelCase(mapObjectName.toLowerCase()) + 's'; // e.g. markers
    this.map[groupName] = this.map[groupName] || [];
    this.map[groupName].push(mapObject);
  }

  removeFromMapObjectGroup(mapObjectName: string, mapObject: any) {
    let groupName = toCamelCase(mapObjectName.toLowerCase()) + 's'; // e.g. markers
    if (this.map && this.map[groupName]) {
      let index = this.map[groupName].indexOf(mapObject);

      (index > -1) && this.map[groupName].splice(index, 1);
    }
  }
}
