import { Component, HostBinding, Inject, Injectable, Input, NgModule, Optional, forwardRef } from '@angular/core';
import { CommonModule } from '@angular/common';
import { config, counter, icon, parse, text } from '@fortawesome/fontawesome-svg-core';
import { DomSanitizer } from '@angular/platform-browser';

/**
 * @fileoverview added by tsickle
 * @suppress {checkTypes} checked by tsc
 */
/**
 * Returns if is IconLookup or not.
 * @return IconLookup
 */
const isIconLookup = (i) => {
    return (/** @type {?} */ (i)).prefix !== undefined && (/** @type {?} */ (i)).iconName !== undefined;
};

/**
 * @fileoverview added by tsickle
 * @suppress {checkTypes} checked by tsc
 */
/**
 * Normalizing icon spec.
 * @return IconLookup
 */
const faNormalizeIconSpec = (iconSpec) => {
    const /** @type {?} */ defaultPrefix = 'fas';
    if (typeof iconSpec === 'undefined' || iconSpec === null) {
        return null;
    }
    if (isIconLookup(iconSpec)) {
        return iconSpec;
    }
    if (Array.isArray(iconSpec) && (/** @type {?} */ (iconSpec)).length === 2) {
        return { prefix: iconSpec[0], iconName: iconSpec[1] };
    }
    if (typeof iconSpec === 'string') {
        return { prefix: defaultPrefix, iconName: iconSpec };
    }
};

/**
 * @fileoverview added by tsickle
 * @suppress {checkTypes} checked by tsc
 */
/**
 * @param T value
 */
const objectWithKey = (key, value) => {
    return (Array.isArray(value) && value.length > 0) || (!Array.isArray(value) && value) ? { [key]: value } : {};
};

/**
 * @fileoverview added by tsickle
 * @suppress {checkTypes} checked by tsc
 */
/**
 * Fontawesome class list.
 * Returns classes array by props.
 * @return string[]
 */
const faClassList = (props) => {
    const /** @type {?} */ classes = {
        'fa-spin': props.spin,
        'fa-pulse': props.pulse,
        'fa-fw': props.fixedWidth,
        'fa-border': props.border,
        'fa-li': props.listItem,
        'fa-inverse': props.inverse,
        'fa-layers-counter': props.counter,
        'fa-flip-horizontal': props.flip === 'horizontal' || props.flip === 'both',
        'fa-flip-vertical': props.flip === 'vertical' || props.flip === 'both',
        [`fa-${props.size}`]: props.size !== null,
        [`fa-rotate-${props.rotate}`]: props.rotate !== null,
        [`fa-pull-${props.pull}`]: props.pull !== null
    };
    return Object.keys(classes)
        .map(key => (classes[key] ? key : null))
        .filter(key => key);
};

/**
 * @fileoverview added by tsickle
 * @suppress {checkTypes} checked by tsc
 */
const faWarnIfIconHtmlMissing = (iconObj, iconSpec) => {
    if (iconSpec && !iconObj) {
        console.error(`FontAwesome: Could not find icon with iconName=${iconSpec.iconName} and prefix=${iconSpec.prefix}`);
    }
};

/**
 * @fileoverview added by tsickle
 * @suppress {checkTypes} checked by tsc
 */
const faWarnIfIconSpecMissing = (iconSpec) => {
    if (!iconSpec) {
        console.error('FontAwesome: Could not find icon. ' +
            `It looks like you've provided a null or undefined icon object to this component.`);
    }
};

/**
 * @fileoverview added by tsickle
 * @suppress {checkTypes} checked by tsc
 */
const faNotFoundIconHtml = `<svg class="${config.replacementClass}" viewBox="0 0 448 512"></svg><!--icon not found-->`;

/**
 * @fileoverview added by tsickle
 * @suppress {checkTypes} checked by tsc
 */
/**
 * Fontawesome icon.
 */
class FaIconComponent {
    /**
     * @param {?} sanitizer
     */
    constructor(sanitizer) {
        this.sanitizer = sanitizer;
        this.classes = [];
    }
    /**
     * @param {?} changes
     * @return {?}
     */
    ngOnChanges(changes) {
        if (changes) {
            this.updateIconSpec();
            this.updateParams();
            this.updateIcon();
            this.renderIcon();
        }
    }
    /**
     * Updating icon spec.
     * @return {?}
     */
    updateIconSpec() {
        this.iconSpec = faNormalizeIconSpec(this.iconProp);
    }
    /**
     * Updating params by component props.
     * @return {?}
     */
    updateParams() {
        const /** @type {?} */ classOpts = {
            flip: this.flip,
            spin: this.spin,
            pulse: this.pulse,
            border: this.border,
            inverse: this.inverse,
            listItem: this.listItem,
            size: this.size || null,
            pull: this.pull || null,
            rotate: this.rotate || null,
            fixedWidth: this.fixedWidth
        };
        const /** @type {?} */ classes = objectWithKey('classes', [...faClassList(classOpts), ...this.classes]);
        const /** @type {?} */ mask = objectWithKey('mask', faNormalizeIconSpec(this.mask));
        const /** @type {?} */ parsedTransform = typeof this.transform === 'string' ? parse.transform(this.transform) : this.transform;
        const /** @type {?} */ transform = objectWithKey('transform', parsedTransform);
        this.params = Object.assign({ title: this.title }, transform, classes, mask, { styles: this.styles, symbol: this.symbol });
    }
    /**
     * Updating icon by params and icon spec.
     * @return {?}
     */
    updateIcon() {
        this.icon = icon(this.iconSpec, this.params);
    }
    /**
     * Rendering icon.
     * @return {?}
     */
    renderIcon() {
        faWarnIfIconSpecMissing(this.iconSpec);
        faWarnIfIconHtmlMissing(this.icon, this.iconSpec);
        this.renderedIconHTML = this.sanitizer.bypassSecurityTrustHtml(this.icon ? this.icon.html.join('\n') : faNotFoundIconHtml);
    }
}
FaIconComponent.decorators = [
    { type: Component, args: [{
                selector: 'fa-icon',
                template: ``,
                host: {
                    class: 'ng-fa-icon',
                }
            },] },
];
/** @nocollapse */
FaIconComponent.ctorParameters = () => [
    { type: DomSanitizer, },
];
FaIconComponent.propDecorators = {
    "renderedIconHTML": [{ type: HostBinding, args: ['innerHTML',] },],
    "iconProp": [{ type: Input, args: ['icon',] },],
    "title": [{ type: Input },],
    "spin": [{ type: Input },],
    "pulse": [{ type: Input },],
    "mask": [{ type: Input },],
    "styles": [{ type: Input },],
    "flip": [{ type: Input },],
    "size": [{ type: Input },],
    "pull": [{ type: Input },],
    "border": [{ type: Input },],
    "inverse": [{ type: Input },],
    "symbol": [{ type: Input },],
    "listItem": [{ type: Input },],
    "rotate": [{ type: Input },],
    "fixedWidth": [{ type: Input },],
    "classes": [{ type: Input },],
    "transform": [{ type: Input },],
};

/**
 * @fileoverview added by tsickle
 * @suppress {checkTypes} checked by tsc
 */
/**
 * Fontawesome layers.
 */
class FaLayersComponent {
}
FaLayersComponent.decorators = [
    { type: Component, args: [{
                selector: 'fa-layers',
                template: `<ng-content select="fa-icon, fa-layers-text, fa-layers-counter"></ng-content>`,
                host: {
                    class: 'fa-layers'
                }
            },] },
];
/** @nocollapse */
FaLayersComponent.ctorParameters = () => [];

/**
 * @fileoverview added by tsickle
 * @suppress {checkTypes} checked by tsc
 */
/**
 * Warns if parent component not existing.
 */
const faWarnIfParentNotExist = (parent, parentName, childName) => {
    if (!parent) {
        console.error(`FontAwesome: ${childName} should be used as child of ${parentName} only.`);
    }
};

/**
 * @fileoverview added by tsickle
 * @suppress {checkTypes} checked by tsc
 */
/**
 * @abstract
 */
class FaLayersTextBaseComponent {
    /**
     * @param {?} parent
     * @param {?} sanitizer
     */
    constructor(parent, sanitizer) {
        this.parent = parent;
        this.sanitizer = sanitizer;
        this.classes = [];
        faWarnIfParentNotExist(this.parent, 'FaLayersComponent', this.constructor.name);
    }
    /**
     * @param {?} changes
     * @return {?}
     */
    ngOnChanges(changes) {
        if (changes) {
            this.updateParams();
            this.updateContent();
        }
    }
    /**
     * Updating content by params and content.
     * @return {?}
     */
    updateContent() {
        this.renderedHTML = this.sanitizer.bypassSecurityTrustHtml(this.renderFontawesomeObject(this.content || '', this.params).html.join('\n'));
    }
}
FaLayersTextBaseComponent.decorators = [
    { type: Injectable },
];
/** @nocollapse */
FaLayersTextBaseComponent.ctorParameters = () => [
    { type: FaLayersComponent, decorators: [{ type: Inject, args: [forwardRef(() => FaLayersComponent),] }, { type: Optional },] },
    { type: DomSanitizer, },
];
FaLayersTextBaseComponent.propDecorators = {
    "renderedHTML": [{ type: HostBinding, args: ['innerHTML',] },],
    "content": [{ type: Input },],
    "title": [{ type: Input },],
    "styles": [{ type: Input },],
    "classes": [{ type: Input },],
};

/**
 * @fileoverview added by tsickle
 * @suppress {checkTypes} checked by tsc
 */
/**
 * Fontawesome layers text.
 */
class FaLayersTextComponent extends FaLayersTextBaseComponent {
    /**
     * Updating params by component props.
     * @return {?}
     */
    updateParams() {
        const /** @type {?} */ classOpts = {
            flip: this.flip,
            spin: this.spin,
            pulse: this.pulse,
            border: this.border,
            inverse: this.inverse,
            listItem: this.listItem,
            size: this.size || null,
            pull: this.pull || null,
            rotate: this.rotate || null,
            fixedWidth: this.fixedWidth
        };
        const /** @type {?} */ classes = objectWithKey('classes', [...faClassList(classOpts), ...this.classes]);
        const /** @type {?} */ parsedTransform = typeof this.transform === 'string' ? parse.transform(this.transform) : this.transform;
        const /** @type {?} */ transform = objectWithKey('transform', parsedTransform);
        this.params = Object.assign({}, transform, classes, { title: this.title, styles: this.styles });
    }
    /**
     * @param {?} content
     * @param {?=} params
     * @return {?}
     */
    renderFontawesomeObject(content, params) {
        return text(content, params);
    }
}
FaLayersTextComponent.decorators = [
    { type: Component, args: [{
                selector: 'fa-layers-text',
                template: '',
                host: {
                    class: 'ng-fa-layers-text'
                }
            },] },
];
/** @nocollapse */
FaLayersTextComponent.ctorParameters = () => [];
FaLayersTextComponent.propDecorators = {
    "spin": [{ type: Input },],
    "pulse": [{ type: Input },],
    "flip": [{ type: Input },],
    "size": [{ type: Input },],
    "pull": [{ type: Input },],
    "border": [{ type: Input },],
    "inverse": [{ type: Input },],
    "listItem": [{ type: Input },],
    "rotate": [{ type: Input },],
    "fixedWidth": [{ type: Input },],
    "transform": [{ type: Input },],
};

/**
 * @fileoverview added by tsickle
 * @suppress {checkTypes} checked by tsc
 */
/**
 * Fontawesome layers counter.
 */
class FaLayersCounterComponent extends FaLayersTextBaseComponent {
    /**
     * Updating params by component props.
     * @return {?}
     */
    updateParams() {
        this.params = {
            title: this.title,
            classes: this.classes,
            styles: this.styles,
        };
    }
    /**
     * @param {?} content
     * @param {?=} params
     * @return {?}
     */
    renderFontawesomeObject(content, params) {
        return counter(content, params);
    }
}
FaLayersCounterComponent.decorators = [
    { type: Component, args: [{
                selector: 'fa-layers-counter',
                template: '',
                host: {
                    class: 'ng-fa-layers-counter'
                }
            },] },
];
/** @nocollapse */
FaLayersCounterComponent.ctorParameters = () => [];

/**
 * @fileoverview added by tsickle
 * @suppress {checkTypes} checked by tsc
 */
class FontAwesomeModule {
}
FontAwesomeModule.decorators = [
    { type: NgModule, args: [{
                imports: [CommonModule],
                declarations: [
                    FaIconComponent,
                    FaLayersComponent,
                    FaLayersTextComponent,
                    FaLayersCounterComponent
                ],
                exports: [
                    FaIconComponent,
                    FaLayersComponent,
                    FaLayersTextComponent,
                    FaLayersCounterComponent
                ],
            },] },
];
/** @nocollapse */
FontAwesomeModule.ctorParameters = () => [];

/**
 * @fileoverview added by tsickle
 * @suppress {checkTypes} checked by tsc
 */

/**
 * @fileoverview added by tsickle
 * @suppress {checkTypes} checked by tsc
 */
/**
 * Generated bundle index. Do not edit.
 */

export { FaIconComponent, FaLayersComponent, FaLayersTextComponent, FaLayersCounterComponent, FontAwesomeModule, FaLayersTextBaseComponent as ɵa };
//# sourceMappingURL=angular-fontawesome.js.map
